package ru.yandex.direct.useractionlog.reader;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.useractionlog.ChangeSource;
import ru.yandex.direct.useractionlog.reader.model.InputCategory;
import ru.yandex.direct.useractionlog.reader.model.OutputCategory;
import ru.yandex.direct.useractionlog.schema.ObjectPath;

public class UserActionLogFilter {
    private LocalDateTime dateFrom;
    private LocalDateTime dateTo;
    private Collection<ObjectPath> objectPaths = new HashSet<>();
    private Collection<InputCategory> categories = new HashSet<>();
    private Collection<OutputCategory> withoutCategories = new HashSet<>();
    private boolean categoriesSpecified;
    private Collection<Long> uids = new HashSet<>();
    private boolean uidsSpecified;
    private Collection<ChangeSource> changeSources = EnumSet.noneOf(ChangeSource.class);
    private boolean changeSourcesSpecified;
    private boolean isInternal;

    public LocalDateTime getDateFrom() {
        return dateFrom;
    }

    public UserActionLogFilter withDateFrom(LocalDateTime dateFrom) {
        this.dateFrom = dateFrom;
        return this;
    }

    public LocalDateTime getDateTo() {
        return dateTo;
    }

    public UserActionLogFilter withDateTo(LocalDateTime dateTo) {
        this.dateTo = dateTo;
        return this;
    }

    public Collection<ObjectPath> getObjectPaths() {
        return objectPaths;
    }

    public UserActionLogFilter withPath(ObjectPath path) {
        objectPaths.add(path);
        return this;
    }

    public Collection<OutputCategory> getOutputCategories() {
        Set<OutputCategory> result;
        if (categoriesSpecified) {
            result = categories.stream()
                    .flatMap(c -> c.toOutputCategories().stream())
                    .collect(Collectors.toSet());
        } else {
            result = EnumSet.allOf(OutputCategory.class);
        }

        result.removeAll(withoutCategories);
        return result;
    }

    public UserActionLogFilter withoutCategories(Collection<OutputCategory> categories) {
        this.withoutCategories = categories;
        return this;
    }

    public UserActionLogFilter withCategories(Collection<InputCategory> categories) {
        this.categories.addAll(categories);
        categoriesSpecified = true;
        return this;
    }

    public boolean isCategoriesSpecified() {
        return categoriesSpecified;
    }

    public UserActionLogFilter withOperatorUids(Collection<Long> uids) {
        this.uids.addAll(uids);
        uidsSpecified = true;
        return this;
    }

    public Collection<Long> getOperatorUids() {
        return uids;
    }

    public boolean isUidsSpecified() {
        return uidsSpecified;
    }

    public UserActionLogFilter withChangeSources(Collection<ChangeSource> changeSources) {
        this.changeSources.addAll(changeSources);
        changeSourcesSpecified = true;
        return this;
    }

    public Collection<ChangeSource> getChangeSources() {
        if (changeSourcesSpecified) {
            return changeSources;
        } else {
            return EnumSet.allOf(ChangeSource.class);
        }
    }

    public boolean isChangeSourcesSpecified() {
        return changeSourcesSpecified;
    }

    public boolean isInternal() {
        return isInternal;
    }

    public void setInternal(boolean internal) {
        isInternal = internal;
    }

    public UserActionLogFilter withInternal(boolean internal) {
        isInternal = internal;
        return this;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UserActionLogFilter that = (UserActionLogFilter) o;
        return categoriesSpecified == that.categoriesSpecified &&
                uidsSpecified == that.uidsSpecified &&
                Objects.equals(dateFrom, that.dateFrom) &&
                Objects.equals(dateTo, that.dateTo) &&
                Objects.equals(objectPaths, that.objectPaths) &&
                Objects.equals(categories, that.categories) &&
                Objects.equals(uids, that.uids);
    }

    @Override
    public int hashCode() {
        return Objects.hash(dateFrom, dateTo, objectPaths, categories, categoriesSpecified, uids, uidsSpecified);
    }
}
