package ru.yandex.direct.useractionlog.reader;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Base64;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;
import ru.yandex.direct.utils.JsonUtils;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Смещение, которое будучи закодированным передаётся пользователю. Далее пользователь должен передать его обратно для
 * получения следующей страницы, никак не модифицируя этот объект.
 */
@JsonIgnoreProperties
@ParametersAreNonnullByDefault
public class UserActionLogOffset implements SplittingIteratorWindow.Offset, ReadActionLogTable.Offset {
    @JsonProperty("t")
    private final long timestamp;

    @JsonProperty("g")
    private final String gtid;

    @JsonProperty("q")
    private final int querySerial;

    @JsonProperty("r")
    private final int rowSerial;

    @JsonProperty("o")
    private final int recordOffset;

    @JsonCreator
    public UserActionLogOffset(
            @JsonProperty("t") long timestamp,
            @JsonProperty("g") String gtid,
            @JsonProperty("q") int querySerial,
            @JsonProperty("r") int rowSerial,
            @JsonProperty("o") int recordOffset) {
        this.timestamp = timestamp;
        this.gtid = gtid;
        this.querySerial = querySerial;
        this.rowSerial = rowSerial;
        this.recordOffset = recordOffset;
    }

    public static UserActionLogOffset fromActionLogRecordWithStats(
            ActionLogRecordWithStats actionLogRecordWithStats, int recordOffset) {
        ActionLogRecord actionLogRecord = actionLogRecordWithStats.getRecord();
        return new UserActionLogOffset(actionLogRecord.getDateTime().toEpochSecond(ZoneOffset.UTC),
                actionLogRecord.getGtid(),
                actionLogRecord.getQuerySerial(),
                actionLogRecord.getRowSerial(),
                recordOffset);
    }

    public static UserActionLogOffset fromToken(String token) {
        return JsonUtils.fromJson(new String(Base64.getDecoder().decode(token), UTF_8), UserActionLogOffset.class);
    }

    @Override
    public int inRecordOffset() {
        return recordOffset;
    }

    public String toToken() {
        byte[] bytes = JsonUtils.toJson(this).getBytes(UTF_8);
        return new String(Base64.getEncoder().encode(bytes), UTF_8);
    }

    @Override
    public LocalDateTime getDateTime() {
        return LocalDateTime.ofEpochSecond(timestamp, 0, ZoneOffset.UTC);
    }

    @Override
    public String getGtid() {
        return gtid;
    }

    @Override
    public int getQuerySerial() {
        return querySerial;
    }

    @Override
    public int getRowSerial() {
        return rowSerial;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UserActionLogOffset offset = (UserActionLogOffset) o;
        return timestamp == offset.timestamp &&
                querySerial == offset.querySerial &&
                rowSerial == offset.rowSerial &&
                recordOffset == offset.recordOffset &&
                Objects.equals(gtid, offset.gtid);
    }

    @Override
    public int hashCode() {
        return Objects.hash(timestamp, gtid, querySerial, rowSerial, recordOffset);
    }
}
