package ru.yandex.direct.useractionlog.reader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.PeekingIterator;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.db.ReadPpclogApiTable;
import ru.yandex.direct.useractionlog.reader.generator.FieldKeyValue;
import ru.yandex.direct.useractionlog.reader.generator.FieldKeyValueUtil;
import ru.yandex.direct.useractionlog.reader.generator.FilterCategories;
import ru.yandex.direct.useractionlog.reader.generator.FilterChangeSource;
import ru.yandex.direct.useractionlog.reader.generator.LogRecordGenerator;
import ru.yandex.direct.useractionlog.reader.generator.LogRecordGeneratorByType;
import ru.yandex.direct.useractionlog.reader.model.LogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;
import ru.yandex.direct.useractionlog.schema.ActionLogSchema;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;

/**
 * Бизнес-логика пользовательских логов, отделённая от слоя пользовательского интерфейса в виде Spring.
 */
@ParametersAreNonnullByDefault
public class UserActionLogReader {
    /**
     * Для постраничной выдачи результата важно, чтобы для фиксированного набора данных один и тот же SQL-запрос всегда
     * выдавал одни и те же записи в одном и том же порядке. Это условие удовлетворяется, если сортировать результат по
     * всем колонкам первичного ключа. Также подразумевается, что журнал событий отсортирован по времени происхождения
     * события.
     */
    private static final Logger logger = LoggerFactory.getLogger(UserActionLogReader.class);
    private static final int MIN_CLICKHOUSE_LIMIT = 400;
    private static final int MAX_CLICKHOUSE_LIMIT = 400;
    private static final int RECORD_STATS_CHUNK_SIZE = 200;
    private final ReadActionLogTable readActionLogTable;
    private final ReadPpclogApiTable readPpclogApiTable;

    public UserActionLogReader(ReadActionLogTable readActionLogTable, ReadPpclogApiTable readPpclogApiTable) {
        this.readActionLogTable = readActionLogTable;
        this.readPpclogApiTable = readPpclogApiTable;
    }

    private static void applyFilter(LogRecordGenerator logRecordGenerator, UserActionLogFilter filter,
                                    SqlBuilder sqlBuilder) {
        if (filter.getDateFrom() != null) {
            sqlBuilder
                    .where(ActionLogSchema.DATE.getExpr() + " >= ?",
                            ActionLogSchema.DATE.getType().toSqlObject(filter.getDateFrom().toLocalDate()))
                    .where(ActionLogSchema.DATETIME.getExpr() + " >= ?",
                            ActionLogSchema.DATETIME.getType().toSqlObject(filter.getDateFrom()));
        }
        if (filter.getDateTo() != null) {
            sqlBuilder
                    .where(ActionLogSchema.DATE.getExpr() + " <= ?",
                            ActionLogSchema.DATE.getType().toSqlObject(filter.getDateTo().toLocalDate()))
                    .where(ActionLogSchema.DATETIME.getExpr() + " <= ?",
                            ActionLogSchema.DATETIME.getType().toSqlObject(filter.getDateTo()));
        }
        if (filter.isUidsSpecified()) {
            if (filter.getOperatorUids().isEmpty()) {
                // Явно указан пустой список uid. По такому запросу ничего не может быть возвращено.
                sqlBuilder.where("0");
            } else {
                sqlBuilder.whereIn(SqlBuilder.column(ActionLogSchema.OPERATOR_UID.getName()), filter.getOperatorUids());
            }
        }
        Map<String, Collection<FieldKeyValue>> fieldValueByTable;
        if (filter.isCategoriesSpecified()) {
            fieldValueByTable = logRecordGenerator.getActionLogRecordTypeToFields(filter.getOutputCategories());
        } else {
            fieldValueByTable = logRecordGenerator.getSupportedActionLogRecordTypeToFields();
        }
        if (fieldValueByTable.isEmpty()) {
            sqlBuilder.where("0");
        } else {
            List<String> values = new ArrayList<>();
            List<String> entryQueries = new ArrayList<>();

            for (Map.Entry<String, Collection<FieldKeyValue>> entry : fieldValueByTable.entrySet()) {
                StringBuilder entryQuery = new StringBuilder();
                entryQuery.append(ActionLogSchema.TYPE.getExpr()).append(" = ?");
                values.add(entry.getKey());

                if (!entry.getValue().isEmpty()) {
                    entryQuery.append(" AND ");
                    Pair<String, List<String>> fieldQuery = FieldKeyValueUtil.toSqlQuery(entry.getValue());
                    entryQuery.append(fieldQuery.getLeft());
                    values.addAll(fieldQuery.getRight());
                }

                entryQueries.add(entryQuery.toString());
            }
            sqlBuilder.where(String.format("(%s)", String.join(" OR ", entryQueries)), values.toArray());
        }

        if (filter.getObjectPaths().isEmpty()) {
            if (!filter.isInternal()) {
                sqlBuilder.where("0");
            }
        } else {
            ReadActionLogTable.objectPathSqlBuilder(sqlBuilder, filter.getObjectPaths());
        }
    }

    /**
     * Список событий, соответствующий заданным критериям.
     *
     * @param filter Критерии
     */
    public FilterResult filterActionLog(UserActionLogFilter filter,
                                        int limit,
                                        @Nullable UserActionLogOffset window,
                                        ReadActionLogTable.Order order,
                                        @Nullable Function<LogRecordGenerator, LogRecordGenerator> alteringLogRecordGeneratorFactory) {
        ReadRequestStats readRequestStats = new ReadRequestStats();
        final LogRecordGenerator baseLogRecordGenerator = new LogRecordGeneratorByType(order);
        LogRecordGenerator endpointLogRecordGenerator = new FilterChangeSource(
                new FilterCategories(baseLogRecordGenerator, filter.getOutputCategories()),
                filter.getChangeSources());
        if (alteringLogRecordGeneratorFactory != null) {
            endpointLogRecordGenerator = alteringLogRecordGeneratorFactory.apply(endpointLogRecordGenerator);
        }

        PeekingIterator<ActionLogRecordWithStats> userActionLogIterator = new FillRecordStats(
                RECORD_STATS_CHUNK_SIZE,
                new FilterByVersion(
                        new DateSplittingActionLogReader(
                                readActionLogTable,
                                OrderedChunkActionLogReader.factoryForFilter(
                                        readActionLogTable,
                                        MAX_CLICKHOUSE_LIMIT,
                                        readRequestStats),
                                sqlBuilder -> applyFilter(baseLogRecordGenerator, filter, sqlBuilder),
                                MIN_CLICKHOUSE_LIMIT,
                                window,
                                order,
                                readRequestStats)),
                readPpclogApiTable,
                readRequestStats);

        SplittingIteratorWindow<ActionLogRecordWithStats, LogRecord, UserActionLogOffset> splittingIteratorWindow =
                new SplittingIteratorWindow<>(
                        endpointLogRecordGenerator::offer,
                        UserActionLogOffset::fromActionLogRecordWithStats);
        MonotonicTime startTime = NanoTimeClock.now();
        SplittingIteratorWindow.Result<LogRecord, UserActionLogOffset> result =
                splittingIteratorWindow.process(userActionLogIterator, limit, window);
        MonotonicTime endTime = NanoTimeClock.now();
        logger.info("Generated {} log records from {} ActionLogRecords by {} queries to user_action_log and " +
                        "{} queries to ppclog_api in {} seconds",
                result.getObjects().size(),
                readRequestStats.recordsFetched,
                readRequestStats.recordQueriesDone + readRequestStats.dateCountQueriesDone,
                readRequestStats.ppclogApiQueriesDone,
                endTime.minus(startTime).toMillis() / 1e3);
        return new FilterResult(result.getObjects(), result.getNextPageOffset());
    }

    /**
     * Объект, группирующий результаты поиска событий с заданным фильтром.
     */
    public static class FilterResult {
        private List<LogRecord> records;
        private UserActionLogOffset offset;

        FilterResult(List<LogRecord> records, @Nullable UserActionLogOffset offset) {
            this.records = records;
            this.offset = offset;
        }

        /**
         * Список событий, соответствующих фильтру
         */
        public List<LogRecord> getRecords() {
            return records;
        }

        /**
         * Если событий больше, чем было запрошено, то возвращается объект, задающий смещение для следующей пачки
         * событий, иначе null.
         */
        @Nullable
        public UserActionLogOffset getOffset() {
            return offset;
        }
    }
}
