package ru.yandex.direct.useractionlog.reader.generator

import com.google.common.collect.PeekingIterator
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList
import ru.yandex.direct.core.entity.bids.service.BidBaseOpt
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory
import ru.yandex.direct.dbschema.ppc.Ppc.PPC
import ru.yandex.direct.useractionlog.reader.model.AdGroupImpressionConditionEvent
import ru.yandex.direct.useractionlog.reader.model.LogRecord
import ru.yandex.direct.useractionlog.reader.model.OutputCategory
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats
import ru.yandex.direct.useractionlog.schema.ObjectPath
import java.util.Optional

class AdGroupImpressionConditionGenerator : DefaultRecordGenerator() {
    companion object {
        private val TABLE_BY_CATEGORY = mapOf(OutputCategory.ADGROUP_IMPRESSION_CONDITION to PPC.BIDS_BASE.name)
        private val FIELD_KEY_VALUES =
            listOf(
                FieldKeyValue.of(PPC.BIDS_BASE.OPTS.name, Optional.empty()),
                FieldKeyValue.of(PPC.BIDS_BASE.RELEVANCE_MATCH_CATEGORIES.name, Optional.empty())
            )
        private val KNOWN_BIDS_BASE_OPTS_VALUES = BidBaseOpt.values().map { it.typedValue }.toSet()
        private val KNOWN_RELEVANCE_MATCH_CATEGORIES_VALUES = RelevanceMatchCategory.values().map { it.name }.toSet()
        private val FIELD_VALUES_BY_CATEGORY = mapOf(OutputCategory.ADGROUP_IMPRESSION_CONDITION to FIELD_KEY_VALUES)
    }

    override fun offer(recordIter: PeekingIterator<ActionLogRecordWithStats>): List<LogRecord> =
        if (recordIter.hasNext()) {
            parseLogRecord(recordIter.next())?.let { listOf(it) } ?: emptyList()
        } else {
            emptyList()
        }

    override fun getTableByCategory(): Map<OutputCategory, String> = TABLE_BY_CATEGORY

    override fun getFieldsValuesByCategory(): Map<OutputCategory, List<FieldKeyValue>> =
        FIELD_VALUES_BY_CATEGORY

    private fun parseLogRecord(record: ActionLogRecordWithStats): LogRecord? =
        parseAdGroupImpression(record)?.let {
            if (it.newState != it.oldState) {
                val r = record.record
                val operatorUid = if (r.directTraceInfo.operatorUid.isPresent) {
                    r.directTraceInfo.operatorUid.asLong
                } else {
                    null
                }
                LogRecord(
                    r.dateTime,
                    operatorUid,
                    r.gtid,
                    it,
                    record.stats.changeSource,
                    record.stats.isChangedByRecommendation
                )
            } else {
                null
            }
        }

    private fun parseAdGroupImpression(record: ActionLogRecordWithStats): AdGroupImpressionConditionEvent? =
        (record.record.path as? ObjectPath.AdGroupPath)?.let { path ->
            AdGroupImpressionConditionEvent(
                path.id.toLong(),
                path.campaignId.toLong(),
                path.clientId.toLong(),
                parseState(record.record.oldFields),
                parseState(record.record.newFields)
            )
        }

    private fun parseState(fieldValueList: FieldValueList): AdGroupImpressionConditionEvent.EventState =
        AdGroupImpressionConditionEvent.EventState(
            parseEnumSet(fieldValueList, PPC.BIDS_BASE.OPTS.name)
                .filter(KNOWN_BIDS_BASE_OPTS_VALUES::contains)
                .map(BidBaseOpt::fromTypedValue)
                .toSet(),
            parseEnumSet(fieldValueList, PPC.BIDS_BASE.RELEVANCE_MATCH_CATEGORIES.name)
                .filter(KNOWN_RELEVANCE_MATCH_CATEGORIES_VALUES::contains)
                .map(RelevanceMatchCategory::fromTypedValue)
                .toSet()
        )

    private fun parseEnumSet(fieldValueList: FieldValueList, name: String): List<String> =
        extractFields(fieldValueList, name)?.split(",") ?: listOf()

    private fun extractFields(fieldValueList: FieldValueList, name: String): String? =
        fieldValueList.fieldsValues.find { name == it.name }
            ?.value
}
