package ru.yandex.direct.useractionlog.reader.generator;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.PeekingIterator;

import ru.yandex.direct.binlogclickhouse.schema.FieldValue;
import ru.yandex.direct.binlogclickhouse.schema.FieldValueList;
import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.reader.model.LogRecord;
import ru.yandex.direct.useractionlog.reader.model.OutputCategory;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;
import ru.yandex.direct.useractionlog.schema.ObjectPath;
import ru.yandex.direct.useractionlog.schema.Operation;
import ru.yandex.misc.lang.StringUtils;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

@ParametersAreNonnullByDefault
public class AdGroupLogRecordGenerator extends DefaultRecordGenerator {
    private static final Duration DURATION_BETWEEN_NEIGHBOURS = Duration.ofSeconds(10);
    private static final int MAX_GROUPED_RECORDS = 100;

    private static final ImmutableMap<OutputCategory, String>
            TABLE_BY_CATEGORY = ImmutableMap.of(OutputCategory.ADGROUP_REGIONS, PPC.PHRASES.getName());
    private static final ImmutableMap<OutputCategory, Collection<FieldKeyValue>> FIELD_VALUES_BY_CATEGORY =
            ImmutableMap.of(
                    OutputCategory.ADGROUP_REGIONS,
                    ImmutableList.of(FieldKeyValue.of(PPC.PHRASES.GEO.getName(), Optional.empty())));
    private final ReadActionLogTable.Order order;

    public AdGroupLogRecordGenerator(ReadActionLogTable.Order order) {
        this.order = order;
    }

    private static List<Integer> parseRegions(String regions) {
        return Stream.of(StringUtils.defaultString(regions).split(","))
                .filter(s -> !s.isEmpty())
                .map(Integer::parseInt)
                .collect(Collectors.toList());
    }

    @Override
    protected Map<OutputCategory, String> getTableByCategory() {
        return TABLE_BY_CATEGORY;
    }

    @Override
    protected Map<OutputCategory, Collection<FieldKeyValue>> getFieldsValuesByCategory() {
        return FIELD_VALUES_BY_CATEGORY;
    }

    @Override
    public List<LogRecord> offer(PeekingIterator<ActionLogRecordWithStats> recordIter) {
        ActionLogRecordWithStats firstRecord = recordIter.next();
        InterestingData previousInterestingData = InterestingData.fromRecord(firstRecord.getRecord()).orElse(null);
        // DIRECT-79799 Если первая запись в пачке не подходит под критерии, то игнорируем её.
        if (previousInterestingData == null) {
            return Collections.emptyList();
        }
        Set<AdGroupId> adGroupIds = new HashSet<>();
        adGroupIds.add(previousInterestingData.path.getId());
        while (recordIter.hasNext() && adGroupIds.size() < MAX_GROUPED_RECORDS) {
            Optional<InterestingData> candidate = InterestingData.fromRecord(recordIter.peek().getRecord());
            if (candidate.isPresent() && candidate.get().sameGroup(previousInterestingData)) {
                previousInterestingData = candidate.get();
                adGroupIds.add(previousInterestingData.path.getId());
                recordIter.next();
            } else {
                break;
            }
        }
        Comparator<AdGroupId> comparator = Comparator.comparingLong(AdGroupId::toLong);
        if (order == ReadActionLogTable.Order.DESC) {
            comparator = comparator.reversed();
        }
        OptionalLong operatorUid = firstRecord.getRecord().getDirectTraceInfo().getOperatorUid();
        return Collections.singletonList(new LogRecord(
                firstRecord.getRecord().getDateTime(),
                operatorUid.isPresent() ? operatorUid.getAsLong() : null,
                firstRecord.getRecord().getGtid(),
                new AdGroupsRegionsEvent(previousInterestingData.path.getClientId(),
                        previousInterestingData.path.getCampaignId(),
                        adGroupIds.stream()
                                .sorted(comparator)
                                .collect(Collectors.toList()),
                        parseRegions(previousInterestingData.oldRegions),
                        parseRegions(previousInterestingData.newRegions)),
                firstRecord.getStats().getChangeSource(),
                firstRecord.getStats().getIsChangedByRecommendation()));
    }

    private static class InterestingData {
        final LocalDateTime dateTime;
        final ObjectPath.AdGroupPath path;
        final String oldRegions;
        final String newRegions;

        private InterestingData(LocalDateTime dateTime, ObjectPath.AdGroupPath path, String oldRegions,
                                String newRegions) {
            this.dateTime = dateTime;
            this.path = path;
            this.oldRegions = oldRegions;
            this.newRegions = newRegions;
        }

        static Optional<InterestingData> fromRecord(ActionLogRecord record) {
            if (record.getPath() instanceof ObjectPath.AdGroupPath) {
                String oldRegions = extractRegions(record.getOldFields()).orElse("");
                if (record.getOperation() == Operation.INSERT || !oldRegions.isEmpty()) {
                    String newRegions = extractRegions(record.getNewFields()).orElse(oldRegions);
                    if (!oldRegions.equals(newRegions)) {
                        return Optional.of(new InterestingData(
                                record.getDateTime(),
                                (ObjectPath.AdGroupPath) record.getPath(),
                                oldRegions,
                                newRegions));
                    }
                }
            }
            return Optional.empty();
        }

        private static Optional<String> extractRegions(FieldValueList fieldValueList) {
            return fieldValueList.getFieldsValues().stream()
                    .filter(f -> Objects.equals(f.getName(), PPC.PHRASES.GEO.getName()))
                    .findAny()
                    .map(FieldValue::getValueAsNonNullString);
        }

        boolean sameGroup(InterestingData other) {
            return path.getCampaignPath().equals(other.path.getCampaignPath())
                    && oldRegions.equals(other.oldRegions)
                    && newRegions.equals(other.newRegions)
                    &&
                    Duration.between(dateTime, other.dateTime)
                            .abs()
                            .compareTo(DURATION_BETWEEN_NEIGHBOURS) < 0;
        }
    }
}
