package ru.yandex.direct.useractionlog.reader.generator;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.common.collect.PeekingIterator;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.binlogclickhouse.schema.FieldValue;
import ru.yandex.direct.useractionlog.AbstractId;
import ru.yandex.direct.useractionlog.ChangeSource;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.reader.model.BannersEvent;
import ru.yandex.direct.useractionlog.reader.model.InputCategory;
import ru.yandex.direct.useractionlog.reader.model.LogRecord;
import ru.yandex.direct.useractionlog.reader.model.OutputCategory;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;
import ru.yandex.direct.useractionlog.schema.AdId;
import ru.yandex.direct.useractionlog.schema.ObjectPath;
import ru.yandex.direct.useractionlog.schema.Operation;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;

@ParametersAreNonnullByDefault
public class BannersLogRecordGenerator extends DefaultRecordGenerator {
    private static final Duration DURATION_BETWEEN_NEIGHBOURS = Duration.ofSeconds(10);
    private static final int MAX_GROUPED_RECORDS = 100;
    private static final Map<OutputCategory, String> TABLE_BY_CATEGORY =
            Maps.asMap(InputCategory.BANNERS_STATUS.toOutputCategories(), ignored -> BANNERS.getName());
    private static final ImmutableMap<OutputCategory, Collection<FieldKeyValue>> FIELDS_VALUES_BY_CATEGORY =
            ImmutableMap.<OutputCategory, Collection<FieldKeyValue>>builder()
                    .put(OutputCategory.BANNERS_ARCHIVED,
                            ImmutableList.of(FieldKeyValue.of(BANNERS.STATUS_ARCH.getName(), Optional.empty())))
                    .put(OutputCategory.BANNERS_HIDE,
                            ImmutableList.of(FieldKeyValue.of(BANNERS.STATUS_SHOW.getName(), Optional.empty())))
                    .put(OutputCategory.BANNERS_SHOW,
                            ImmutableList.of(FieldKeyValue.of(BANNERS.STATUS_SHOW.getName(), Optional.empty())))
                    .put(OutputCategory.BANNERS_UNARCHIVED,
                            ImmutableList.of(FieldKeyValue.of(BANNERS.STATUS_ARCH.getName(), Optional.empty())))
                    .build();

    static {
        if (!TABLE_BY_CATEGORY.keySet().equals(FIELDS_VALUES_BY_CATEGORY.keySet())) {
            throw new IllegalStateException("mismatch between specified categories");
        }
    }

    private final ReadActionLogTable.Order order;

    public BannersLogRecordGenerator(ReadActionLogTable.Order order) {
        this.order = order;
    }

    @Nullable
    private static ObjectPath.AdGroupPath groupPathFromAdPath(ObjectPath path) {
        try {
            return (ObjectPath.AdGroupPath) ObjectPath.AdPath.class.cast(path).getAdGroupPath();
        } catch (ClassCastException e) {
            return null;
        }
    }

    /**
     * Берёт из итератора группу записей. В этой группе:
     * <ul>
     * <li>Все записи отсортированы по возрастанию времени, независимо от исходной сортировки.</li>
     * <li>Не более {@link #MAX_GROUPED_RECORDS} записей.</li>
     * <li>Все записи имеют один и тот же {@link ActionLogRecord#getType()}.</li>
     * <li>Все записи относятся к одной и той же группе баннеров.</li>
     * <li>Все записи имеют один и тот же operator uid и change source.</li>
     * <li>Время между каждой соседней записью не более {@link #DURATION_BETWEEN_NEIGHBOURS}.</li>
     * </ul>
     */
    private static List<ActionLogRecordWithStats> nextRecordGroup(
            PeekingIterator<ActionLogRecordWithStats> recordIter) {
        List<ActionLogRecordWithStats> records = new ArrayList<>();
        if (!recordIter.hasNext()) {
            return records;
        }
        ActionLogRecordWithStats candidate = recordIter.peek();
        ObjectPath.AdGroupPath path = groupPathFromAdPath(candidate.getRecord().getPath());
        if (path == null) {
            return records;
        }
        records.add(recordIter.next());
        String type = candidate.getRecord().getType();
        OptionalLong operatorUid = candidate.getRecord().getDirectTraceInfo().getOperatorUid();
        ChangeSource changeSource = candidate.getStats().getChangeSource();
        Boolean isChangedByRecommendation = candidate.getStats().getIsChangedByRecommendation();
        while (recordIter.hasNext()
                && records.size() < MAX_GROUPED_RECORDS
                && recordIter.peek().getRecord().getType().equals(type)
                && recordIter.peek().getRecord().getDirectTraceInfo().getOperatorUid().equals(operatorUid)
                && recordIter.peek().getStats().getChangeSource().equals(changeSource)
                && recordIter.peek().getStats().getIsChangedByRecommendation().equals(isChangedByRecommendation)
                && Objects.equals(groupPathFromAdPath(recordIter.peek().getRecord().getPath()), path)
                &&
                Duration.between(records.get(records.size() - 1).getRecord().getDateTime(),
                        recordIter.peek().getRecord().getDateTime())
                        .abs()
                        .compareTo(DURATION_BETWEEN_NEIGHBOURS) < 0) {
            records.add(recordIter.next());
        }
        // Результат не сортируется, т.к. всё равно он потом будет растаскан по SortedSet с другим компаратором.
        return records;

    }

    @Nonnull
    private static List<LogRecord> mergeCollectionsIntoLogEvents(
            SortedSet<AdId> statusArchivedOn,
            SortedSet<AdId> statusArchivedOff,
            SortedSet<AdId> statusShowOn,
            SortedSet<AdId> statusShowOff,
            List<ActionLogRecordWithStats> recordGroup,
            ActionLogRecordWithStats firstRecord) {
        final ObjectPath.AdGroupPath sharedGroupPath = groupPathFromAdPath(firstRecord.getRecord().getPath());
        final String sharedGtid = firstRecord.getRecord().getGtid();
        final Long sharedOperatorUid = firstRecord.getRecord().getDirectTraceInfo().getOperatorUid().isPresent()
                ? firstRecord.getRecord().getDirectTraceInfo().getOperatorUid().getAsLong()
                : null;
        final ChangeSource sharedChangeSource = firstRecord.getStats().getChangeSource();
        final Boolean isChangedByRecommendation = firstRecord.getStats().getIsChangedByRecommendation();
        List<LogRecord> result = new ArrayList<>();
        List<Pair<Collection<AdId>, OutputCategory>> pairs = Arrays.asList(
                Pair.of(statusArchivedOn, OutputCategory.BANNERS_ARCHIVED),
                Pair.of(statusArchivedOff, OutputCategory.BANNERS_UNARCHIVED),
                Pair.of(statusShowOn, OutputCategory.BANNERS_SHOW),
                Pair.of(statusShowOff, OutputCategory.BANNERS_HIDE));
        for (Pair<Collection<AdId>, OutputCategory> pair : pairs) {
            if (!pair.getLeft().isEmpty()) {
                ObjectPath.AdGroupPath path = Objects.requireNonNull(sharedGroupPath);
                LocalDateTime dateTime = recordGroup.stream()
                        .filter(r -> pair.getLeft().contains((AdId) r.getRecord().getPath().getId()))
                        .map(r -> r.getRecord().getDateTime())
                        .min(Comparator.naturalOrder())
                        .orElseThrow(() -> new IllegalStateException("impossible"));
                result.add(new LogRecord(
                        dateTime,
                        sharedOperatorUid,
                        Objects.requireNonNull(sharedGtid),
                        new BannersEvent(path.getClientId(),
                                path.getCampaignId(),
                                path.getId(),
                                new ArrayList<>(pair.getLeft()),
                                pair.getRight()),
                        sharedChangeSource,
                        isChangedByRecommendation));
            }
        }
        return result;
    }

    private static void putRecordIntoCollections(SortedSet<AdId> statusArchivedOn,
                                                 SortedSet<AdId> statusArchivedOff,
                                                 SortedSet<AdId> statusShowOn,
                                                 SortedSet<AdId> statusShowOff,
                                                 ActionLogRecordWithStats record) {
        // Баннер не может быть создан в архиве и сразу же быть разархивирован.
        // Поэтому при создании баннера не следует реагировать на установку значения разархивации
        // как на реальную разархивацию.
        boolean isInsert = record.getRecord().getOperation() == Operation.INSERT;
        AdId adId = ((ObjectPath.AdPath) record.getRecord().getPath()).getId();
        for (FieldValue<String> field : record.getRecord().getNewFields().getFieldsValues()) {
            Collection<AdId> destination = null;
            if (field.getName().equals(BANNERS.STATUS_ARCH.getName())) {
                if (field.getValue().equalsIgnoreCase("yes")) {
                    destination = statusArchivedOn;
                } else if (!isInsert) {
                    destination = statusArchivedOff;
                }
            } else if (field.getName().equals(BANNERS.STATUS_SHOW.getName())) {
                if (field.getValue().equalsIgnoreCase("yes")) {
                    destination = statusShowOn;
                } else {
                    destination = statusShowOff;
                }
            }
            if (destination != null) {
                destination.add(adId);
            }
        }
    }

    protected Map<OutputCategory, String> getTableByCategory() {
        return TABLE_BY_CATEGORY;
    }

    @Override
    protected Map<OutputCategory, Collection<FieldKeyValue>> getFieldsValuesByCategory() {
        return FIELDS_VALUES_BY_CATEGORY;
    }

    @Override
    public List<LogRecord> offer(PeekingIterator<ActionLogRecordWithStats> recordIter) {
        SortedSet<AdId> statusArchivedOn = new TreeSet<>(Comparator.comparing(AbstractId::toLong));
        SortedSet<AdId> statusArchivedOff = new TreeSet<>(Comparator.comparing(AbstractId::toLong));
        SortedSet<AdId> statusShowOn = new TreeSet<>(Comparator.comparing(AbstractId::toLong));
        SortedSet<AdId> statusShowOff = new TreeSet<>(Comparator.comparing(AbstractId::toLong));
        List<ActionLogRecordWithStats> recordGroup = nextRecordGroup(recordIter);
        if (recordGroup.isEmpty()) {
            return Collections.emptyList();
        }

        for (ActionLogRecordWithStats record : recordGroup) {
            putRecordIntoCollections(statusArchivedOn, statusArchivedOff, statusShowOn, statusShowOff, record);
        }

        List<Pair<Collection<AdId>, Collection<AdId>>> opposites = Arrays.asList(
                Pair.of(statusArchivedOn, statusArchivedOff),
                Pair.of(statusShowOn, statusShowOff));
        for (Pair<Collection<AdId>, Collection<AdId>> opposite : opposites) {
            opposite.getLeft().removeIf(opposite.getRight()::contains);
            opposite.getRight().removeIf(opposite.getLeft()::contains);
        }

        ActionLogRecordWithStats firstRecord = recordGroup.get(0);
        List<LogRecord> result =
                mergeCollectionsIntoLogEvents(statusArchivedOn, statusArchivedOff, statusShowOn, statusShowOff,
                        recordGroup,
                        firstRecord);
        if (order == ReadActionLogTable.Order.DESC) {
            Collections.reverse(result);
        }
        return result;
    }
}
