package ru.yandex.direct.useractionlog.reader.generator;

import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ObjectPath;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;

@ParametersAreNonnullByDefault
class CampaignSingleRecordMetaData extends SingleRecordMetaData {
    private final CampaignId campaignId;
    private final Map<CampaignOpts, BoolTransition> optsChanges;

    CampaignSingleRecordMetaData(ActionLogRecord record) {
        super(record);
        if (record.getPath() instanceof ObjectPath.CampaignPath) {
            this.campaignId = (CampaignId) record.getPath().getId();
        } else if (record.getPath() instanceof ObjectPath.AdGroupPath) {
            this.campaignId = ((ObjectPath.AdGroupPath) record.getPath()).getCampaignId();
        } else if (record.getPath() instanceof ObjectPath.AdPath) {
            this.campaignId = ((ObjectPath.AdPath) record.getPath()).getCampaignId();
        } else {
            throw new IllegalArgumentException("Can't handle object path " + record.getPath().toPathString());
        }
        this.optsChanges = Collections.unmodifiableMap(makeCampaignOptMap(getBeforeFields(), getAfterFields()));
    }

    private static Map<CampaignOpts, BoolTransition> makeCampaignOptMap(
            Map<String, String> beforeFields, Map<String, String> afterFields) {
        Map<CampaignOpts, BoolTransition> whatMap = new EnumMap<>(CampaignOpts.class);
        for (CampaignOpts opt : CampaignOpts.values()) {
            whatMap.put(opt, BoolTransition.UNCHANGED_FALSE);
        }
        Set<CampaignOpts> oldOpts = CampaignMappings.optsFromDb(beforeFields.get(CAMPAIGNS.OPTS.getName()));
        if (oldOpts == null) {
            oldOpts = EnumSet.noneOf(CampaignOpts.class);
        }
        Set<CampaignOpts> newOpts = CampaignMappings.optsFromDb(afterFields.get(CAMPAIGNS.OPTS.getName()));
        if (newOpts == null) {
            // Если поле есть в before и отсутствует в after, значит дедупликатор вырезал его из after,
            // значит оно не изменилось.
            newOpts = oldOpts;
        }
        for (CampaignOpts opt : oldOpts) {
            if (newOpts.contains(opt)) {
                whatMap.put(opt, BoolTransition.UNCHANGED_TRUE);
            } else {
                whatMap.put(opt, BoolTransition.SET_TO_FALSE);
            }
        }
        for (CampaignOpts opt : newOpts) {
            if (!oldOpts.contains(opt)) {
                whatMap.put(opt, BoolTransition.SET_TO_TRUE);
            }
        }
        return whatMap;
    }

    public CampaignId getCampaignId() {
        return campaignId;
    }

    public Map<CampaignOpts, BoolTransition> getOptsChanges() {
        return optsChanges;
    }

    @Nullable
    public BoolTransition yesNoTransition(String fieldName) {
        final boolean before;
        final boolean after;
        String beforeRaw = getBeforeFields().get(fieldName);
        if (beforeRaw == null) {
            return null;
        } else {
            before = beforeRaw.equalsIgnoreCase("yes");
        }
        String afterRaw = getAfterFields().get(fieldName);
        if (afterRaw == null) {
            after = before;
        } else {
            after = afterRaw.equalsIgnoreCase("yes");
        }
        return BoolTransition.get(before, after);
    }
}
