package ru.yandex.direct.useractionlog.reader.generator;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.PeekingIterator;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.jooq.TableField;
import org.jooq.tools.Convert;

import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsDayBudgetShowMode;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyGroup;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName;
import ru.yandex.direct.grid.model.campaign.GdiDayBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;
import ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorFacade;
import ru.yandex.direct.libs.timetarget.HoursCoef;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.libs.timetarget.WeekdayType;
import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.reader.model.BroadMatchView;
import ru.yandex.direct.useractionlog.reader.model.CampaignBroadMatchEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignListChangeEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignNetworkEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignRegionsEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignStatusChangeEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignStrategyEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignTimeTargetEvent;
import ru.yandex.direct.useractionlog.reader.model.CampaignValueChangeEvent;
import ru.yandex.direct.useractionlog.reader.model.LogEvent;
import ru.yandex.direct.useractionlog.reader.model.LogRecord;
import ru.yandex.direct.useractionlog.reader.model.NetworkView;
import ru.yandex.direct.useractionlog.reader.model.NetworkViewType;
import ru.yandex.direct.useractionlog.reader.model.OutputCategory;
import ru.yandex.direct.useractionlog.reader.model.TimeTargetView;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;
import ru.yandex.direct.useractionlog.schema.Operation;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.misc.lang.StringUtils;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.grid.model.entity.campaign.strategy.GdStrategyExtractorHelper.STRATEGIES_EXTRACTORS_BY_TYPES;

@ParametersAreNonnullByDefault
public class CampaignsLogRecordGenerator extends DefaultRecordGenerator {
    private static final String REMOVED = "__removed";
    private static final String ADDED = "__added";
    private final ImmutableMap<String, CaseHandler> caseHandlerByField;
    private final ImmutableMap<OutputCategory, String> tableByCategory;
    private final ImmutableMap<OutputCategory, Collection<FieldKeyValue>> fieldsValuesByCategory;
    private final ReadActionLogTable.Order order;

    public CampaignsLogRecordGenerator(ReadActionLogTable.Order order) {
        this.order = order;
        // Обработчики ниже могут вызываться несколько раз для одной записи.
        // Перед каждым вызовом делается проверка, что этот конкретный обработчик ещё не вызывался
        // для какой-то конкретной записи. Если в хеш-таблицу положить два разных экземпляра
        // одного класса, то пользователь может увидеть два одинаковых события.
        ArchivedCaseHandler archivedCaseHandler = new ArchivedCaseHandler();
        BroadMatchCaseHandler broadMatchCaseHandler = new BroadMatchCaseHandler();
        DisabledDomainsCaseHandler disabledDomainsCaseHandler = new DisabledDomainsCaseHandler();
        DisabledIpsCaseHandler disabledIpsCaseHandler = new DisabledIpsCaseHandler();
        DisabledSspCaseHandler disabledSspCaseHandler = new DisabledSspCaseHandler();
        FinishTimeCaseHandler finishTimeCaseHandler = new FinishTimeCaseHandler();
        MinusWordsCaseHandler minusWordsCaseHandler = new MinusWordsCaseHandler();
        NetworkCaseHandler networkCaseHandler = new NetworkCaseHandler();
        OptsCaseHandler optsCaseHandler = new OptsCaseHandler(networkCaseHandler);
        RegionsCaseHandler regionsCaseHandler = new RegionsCaseHandler();
        RestrictFrequencyCaseHandler restrictFrequencyHandler = new RestrictFrequencyCaseHandler();
        StartTimeCaseHandler startTimeCaseHandler = new StartTimeCaseHandler();
        StatusShowCaseHandler statusShowCaseHandler = new StatusShowCaseHandler();
        StrategyCaseHandler strategyCaseHandler = new StrategyCaseHandler();
        TimeTargetCaseHandler timeTargetCaseHandler = new TimeTargetCaseHandler();

        List<Triple<TableField<?, ?>, String, CaseHandler>> fieldSuffixHandlerList = Arrays.asList(
                Triple.of(CAMPAIGNS.ARCHIVED, "", archivedCaseHandler),
                Triple.of(CAMPAIGNS.ATTRIBUTION_MODEL, "", strategyCaseHandler),
                Triple.of(CAMPAIGNS.AUTOBUDGET, "", strategyCaseHandler),
                Triple.of(CAMPAIGNS.CONTEXT_LIMIT, "", networkCaseHandler),
                Triple.of(CAMPAIGNS.CONTEXT_PRICE_COEF, "", networkCaseHandler),
                Triple.of(CAMPAIGNS.DAY_BUDGET, "", strategyCaseHandler),
                Triple.of(CAMPAIGNS.DAY_BUDGET_SHOW_MODE, "", strategyCaseHandler),
                Triple.of(CAMPAIGNS.DISABLED_IPS, ADDED, disabledIpsCaseHandler),
                Triple.of(CAMPAIGNS.DISABLED_IPS, REMOVED, disabledIpsCaseHandler),
                Triple.of(CAMPAIGNS.DISABLED_SSP, ADDED, disabledSspCaseHandler),
                Triple.of(CAMPAIGNS.DISABLED_SSP, REMOVED, disabledSspCaseHandler),
                Triple.of(CAMPAIGNS.DONT_SHOW, ADDED, disabledDomainsCaseHandler),
                Triple.of(CAMPAIGNS.DONT_SHOW, REMOVED, disabledDomainsCaseHandler),
                Triple.of(CAMPAIGNS.FINISH_TIME, "", finishTimeCaseHandler),
                Triple.of(CAMPAIGNS.GEO, "", regionsCaseHandler),
                Triple.of(CAMPAIGNS.OPTS, "", optsCaseHandler),
                Triple.of(CAMPAIGNS.PLATFORM, "", strategyCaseHandler),
                Triple.of(CAMPAIGNS.RF, "", restrictFrequencyHandler),
                Triple.of(CAMPAIGNS.RF_RESET, "", restrictFrequencyHandler),
                Triple.of(CAMPAIGNS.START_TIME, "", startTimeCaseHandler),
                Triple.of(CAMPAIGNS.STATUS_SHOW, "", statusShowCaseHandler),
                Triple.of(CAMPAIGNS.STRATEGY_DATA, "", strategyCaseHandler),
                Triple.of(CAMPAIGNS.STRATEGY_NAME, "", strategyCaseHandler),
                Triple.of(CAMPAIGNS.TIME_TARGET, "", timeTargetCaseHandler),
                Triple.of(CAMPAIGNS.TIMEZONE_ID, "", timeTargetCaseHandler),
                Triple.of(CAMP_OPTIONS.BROAD_MATCH_FLAG, "", broadMatchCaseHandler),
                Triple.of(CAMP_OPTIONS.BROAD_MATCH_GOAL_ID, "", broadMatchCaseHandler),
                Triple.of(CAMP_OPTIONS.BROAD_MATCH_LIMIT, "", broadMatchCaseHandler),
                Triple.of(CAMP_OPTIONS.BROAD_MATCH_RATE, "", broadMatchCaseHandler),
                Triple.of(CAMP_OPTIONS.MINUS_WORDS, ADDED, minusWordsCaseHandler),
                Triple.of(CAMP_OPTIONS.MINUS_WORDS, REMOVED, minusWordsCaseHandler));

        caseHandlerByField = fieldSuffixHandlerList.stream().collect(ImmutableMap.toImmutableMap(
                fieldSuffixHandler -> fieldSuffixHandler.getLeft().getName() + fieldSuffixHandler.getMiddle(),
                Triple::getRight));

        Map<OutputCategory, String> tableByCategoryBuilder = new EnumMap<>(OutputCategory.class);
        Map<OutputCategory, Set<FieldKeyValue>> fieldsValuesMap = new EnumMap<>(OutputCategory.class);
        for (Triple<TableField<?, ?>, String, CaseHandler> item : fieldSuffixHandlerList) {
            for (OutputCategory outputCategory : item.getRight().getCategories()) {
                tableByCategoryBuilder.put(outputCategory, item.getLeft().getTable().getName());
                fieldsValuesMap
                        .computeIfAbsent(outputCategory, ignored -> new HashSet<>())
                        .add(FieldKeyValue.of(item.getLeft().getName(), Optional.empty()));
            }
        }
        tableByCategoryBuilder.put(OutputCategory.CAMPAIGN_CREATION, CAMPAIGNS.getName());
        fieldsValuesMap.put(OutputCategory.CAMPAIGN_CREATION, Collections.emptySet());
        this.tableByCategory = ImmutableMap.copyOf(tableByCategoryBuilder);
        this.fieldsValuesByCategory = ImmutableMap.copyOf(
                Maps.transformValues(fieldsValuesMap, ImmutableSet::copyOf));
        if (!tableByCategory.keySet().equals(fieldsValuesByCategory.keySet())) {
            throw new IllegalStateException("mismatch between specified categories");
        }
    }

    private static Stream<String> splitCommaDelimited(@Nullable String source) {
        return Stream.of(StringUtils.defaultString(source).split(","))
                .filter(s -> !s.isEmpty());
    }

    private static Optional<Pair<String, String>> extractRemovedAdded(CampaignSingleRecordMetaData owner,
                                                                      String name) {
        switch (owner.getRecord().getOperation()) {
            case INSERT:
                return Optional.of(Pair.of("", StringUtils.defaultString(owner.getAfterFields().get(name))));
            case UPDATE:
                return Optional.of(Pair.of(
                        StringUtils.defaultString(owner.getAfterFields().get(name + REMOVED)),
                        StringUtils.defaultString(owner.getAfterFields().get(name + ADDED))));
            default:
                return Optional.empty();
        }
    }

    @Override
    public List<LogRecord> offer(PeekingIterator<ActionLogRecordWithStats> recordIter) {
        ActionLogRecordWithStats recordWithStats = recordIter.next();
        if (recordWithStats.getRecord().getOperation() == Operation.DELETE) {
            return Collections.emptyList();
        } else {
            CampaignSingleRecordMetaData metaData = new CampaignSingleRecordMetaData(recordWithStats.getRecord());
            RunOnce<CampaignSingleRecordMetaData> acceptor = new RunOnce<>(metaData);
            if (recordWithStats.getRecord().getOperation() == Operation.INSERT &&
                    recordWithStats.getRecord().getType().equals(CAMPAIGNS.getName())) {
                metaData.addEvent(new CampaignStatusChangeEvent(metaData.getClientId(), metaData.getCampaignId(),
                        OutputCategory.CAMPAIGN_CREATION));
            }
            for (String keyFromAfter : metaData.getAfterFields().keySet()) {
                RunOnce.Visitor<CampaignSingleRecordMetaData> caseHandler = caseHandlerByField.get(keyFromAfter);
                if (caseHandler != null) {
                    acceptor.accept(caseHandler);
                }
            }
            List<LogRecord> result = new ArrayList<>(metaData.getEvents().size());
            for (LogEvent event : metaData.getEvents()) {
                OptionalLong operatorUid = recordWithStats.getRecord().getDirectTraceInfo().getOperatorUid();
                result.add(new LogRecord(recordWithStats.getRecord().getDateTime(),
                        operatorUid.isPresent() ? operatorUid.getAsLong() : null,
                        recordWithStats.getRecord().getGtid(),
                        event,
                        recordWithStats.getStats().getChangeSource(),
                        recordWithStats.getStats().getIsChangedByRecommendation()));

            }
            if (order == ReadActionLogTable.Order.DESC) {
                Collections.reverse(result);
            }
            return result;
        }
    }

    @Override
    protected Map<OutputCategory, String> getTableByCategory() {
        return tableByCategory;
    }

    @Override
    protected Map<OutputCategory, Collection<FieldKeyValue>> getFieldsValuesByCategory() {
        return fieldsValuesByCategory;
    }

    private interface CaseHandler extends RunOnce.Visitor<CampaignSingleRecordMetaData> {
        ImmutableSet<OutputCategory> getCategories();
    }

    private static class ArchivedCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            BoolTransition transition = owner.yesNoTransition(CAMPAIGNS.ARCHIVED.getName());
            if (transition != null && transition.isChange) {
                owner.addEvent(new CampaignStatusChangeEvent(owner.getClientId(),
                        owner.getCampaignId(),
                        transition.newValue ? OutputCategory.CAMPAIGN_ARCHIVED : OutputCategory.CAMPAIGN_UNARCHIVED));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_ARCHIVED, OutputCategory.CAMPAIGN_UNARCHIVED);
        }
    }

    private static class BroadMatchCaseHandler implements CaseHandler {
        private BroadMatchView makeView(@Nullable String flag, @Nullable String limit, @Nullable String goalIdRaw) {
            return new BroadMatchView(
                    flag != null && flag.equalsIgnoreCase("yes"),
                    (limit == null || limit.equals("0")) ? null : Integer.parseInt(limit),
                    Optional.ofNullable(goalIdRaw).map(Long::parseLong).orElse(null));
        }

        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            BroadMatchView before;
            BroadMatchView after;
            if (owner.getRecord().getOperation() == Operation.INSERT) {
                before = makeView(null, null, null);
                after = makeView(
                        owner.getAfterFields().get(CAMP_OPTIONS.BROAD_MATCH_FLAG.getName()),
                        owner.getAfterFields().get(CAMP_OPTIONS.BROAD_MATCH_LIMIT.getName()),
                        owner.getAfterFields().get(CAMP_OPTIONS.BROAD_MATCH_GOAL_ID.getName()));
            } else if (owner.getRecord().getOperation() == Operation.UPDATE
                    && owner.getBeforeFields().containsKey(CAMP_OPTIONS.BROAD_MATCH_FLAG.getName())
                    && owner.getBeforeFields().containsKey(CAMP_OPTIONS.BROAD_MATCH_LIMIT.getName())
                    && owner.getBeforeFields().containsKey(CAMP_OPTIONS.BROAD_MATCH_GOAL_ID.getName())) {
                before = makeView(
                        owner.getBeforeFields().get(CAMP_OPTIONS.BROAD_MATCH_FLAG.getName()),
                        owner.getBeforeFields().get(CAMP_OPTIONS.BROAD_MATCH_LIMIT.getName()),
                        owner.getBeforeFields().get(CAMP_OPTIONS.BROAD_MATCH_GOAL_ID.getName()));
                after = makeView(
                        owner.getFromAfterOrBefore(CAMP_OPTIONS.BROAD_MATCH_FLAG.getName()),
                        owner.getFromAfterOrBefore(CAMP_OPTIONS.BROAD_MATCH_LIMIT.getName()),
                        owner.getFromAfterOrBefore(CAMP_OPTIONS.BROAD_MATCH_GOAL_ID.getName()));
            } else {
                before = after = null;
            }

            if (!Objects.equals(before, after)) {
                owner.addEvent(new CampaignBroadMatchEvent(owner.getClientId(),
                        owner.getCampaignId(),
                        Objects.requireNonNull(before),
                        Objects.requireNonNull(after)));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_BROAD_MATCH);
        }
    }

    private static class DisabledDomainsCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            final String name = CAMPAIGNS.DONT_SHOW.getName();
            extractRemovedAdded(owner, name).ifPresent(pair -> {
                List<String> old = splitCommaDelimited(owner.getBeforeFields().get(name))
                        .collect(Collectors.toList());
                List<String> removed = splitCommaDelimited(pair.getLeft())
                        .collect(Collectors.toList());
                List<String> added = splitCommaDelimited(pair.getRight())
                        .collect(Collectors.toList());
                if (!removed.isEmpty() || !added.isEmpty()) {
                    owner.addEvent(new CampaignListChangeEvent(
                            owner.getClientId(),
                            owner.getCampaignId(),
                            OutputCategory.CAMPAIGN_DISABLED_DOMAINS,
                            old,
                            removed,
                            added));
                }
            });
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_DISABLED_DOMAINS);
        }
    }

    private static class DisabledIpsCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            final String name = CAMPAIGNS.DISABLED_IPS.getName();
            extractRemovedAdded(owner, name).ifPresent(pair -> {
                List<String> old = splitCommaDelimited(owner.getBeforeFields().get(name))
                        .collect(Collectors.toList());
                List<String> removed = splitCommaDelimited(pair.getLeft())
                        .collect(Collectors.toList());
                List<String> added = splitCommaDelimited(pair.getRight())
                        .collect(Collectors.toList());
                if (!removed.isEmpty() || !added.isEmpty()) {
                    owner.addEvent(new CampaignListChangeEvent(
                            owner.getClientId(),
                            owner.getCampaignId(),
                            OutputCategory.CAMPAIGN_DISABLED_IPS,
                            old,
                            removed,
                            added));
                }
            });
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_DISABLED_IPS);
        }
    }

    private static class DisabledSspCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            String name = CAMPAIGNS.DISABLED_SSP.getName();
            @SuppressWarnings("unchecked")
            List<String> old = JsonUtils.fromJson(owner.getBeforeFields().get(name), List.class);
            @SuppressWarnings("unchecked")
            List<String> removed = JsonUtils.fromJson(owner.getAfterFields().get(name + REMOVED), List.class);
            @SuppressWarnings("unchecked")
            List<String> added = JsonUtils.fromJson(owner.getAfterFields().get(name + ADDED), List.class);
            if (!removed.isEmpty() || !added.isEmpty()) {
                owner.addEvent(new CampaignListChangeEvent(
                        owner.getClientId(),
                        owner.getCampaignId(),
                        OutputCategory.CAMPAIGN_DISABLED_SSP,
                        old,
                        removed,
                        added));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_DISABLED_SSP);
        }
    }

    private static class FinishTimeCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> forward) {
            String name = CAMPAIGNS.FINISH_TIME.getName();
            LocalDate oldFinishDate =
                    Optional.ofNullable(owner.getBeforeFields().get(name)).map(LocalDate::parse).orElse(null);
            LocalDate newFinishDate =
                    Optional.ofNullable(owner.getAfterFields().get(name)).map(LocalDate::parse).orElse(null);
            if (!Objects.equals(oldFinishDate, newFinishDate)) {
                owner.addEvent(new CampaignValueChangeEvent(owner.getClientId(),
                        owner.getCampaignId(),
                        OutputCategory.CAMPAIGN_FINISH_TIME,
                        Optional.ofNullable(oldFinishDate).map(LocalDate::toString).orElse(null),
                        Optional.ofNullable(newFinishDate).map(LocalDate::toString).orElse(null)));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_FINISH_TIME);
        }
    }

    private static class MinusWordsCaseHandler implements CaseHandler {
        @Override
        @SuppressWarnings("unchecked")
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            final String name = CAMP_OPTIONS.MINUS_WORDS.getName();
            extractRemovedAdded(owner, name).ifPresent(pair -> {
                List<String> oldMinusWords;
                String oldMinusWordsRaw = owner.getBeforeFields().get(name);
                if (StringUtils.isEmpty(oldMinusWordsRaw)) {
                    oldMinusWords = Collections.emptyList();
                } else {
                    oldMinusWords = JsonUtils.fromJson(oldMinusWordsRaw, List.class);
                }
                @SuppressWarnings("unchecked")
                List<String> removedMinusWords = JsonUtils.fromJson(pair.getLeft(), List.class);
                @SuppressWarnings("unchecked")
                List<String> addedMinusWords = JsonUtils.fromJson(pair.getRight(), List.class);
                if (!removedMinusWords.isEmpty() || !addedMinusWords.isEmpty()) {
                    owner.addEvent(new CampaignListChangeEvent(owner.getClientId(),
                            owner.getCampaignId(),
                            OutputCategory.CAMPAIGN_MINUS_WORDS,
                            oldMinusWords,
                            removedMinusWords,
                            addedMinusWords));
                }
            });
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_MINUS_WORDS);
        }
    }

    private static class NetworkCaseHandler implements CaseHandler {
        private static final NetworkView DISABLED = new NetworkView(NetworkViewType.disabled, 0, 0, false);

        private static NetworkView makeNetworkView(String platform, int contextLimit, int contextPriceCoef,
                                                   boolean enableCpcHold) {
            NetworkViewType type;
            // По всему коду пользовательских логов избегается завязывание на enum'ы, максимально возможно
            // используются их строковые представления. Гипотетически в любой enum могут добавиться новые значения,
            // удалиться старые, но из старых логов старые enum'ы никуда не денутся.
            // В случае удаления какого-либо enum'а старое значение можно просто заменить на строковое представление.
            if (platform.equals(ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform.search.getLiteral())) {
                return DISABLED;
            }
            int percent;
            switch (contextLimit) {
                case 0:
                    // Если это значение 0, значит выставленная стратегия конфликтует с
                    // настройками в сетях.
                    return DISABLED;
                case 254:
                case 255:
                    // В документации к схеме БД сказано, что 255 - это отсутствие ограничения.
                    // В бинлоге за три месяца ни разу такого значения не было.
                    // По договорённости в DIRECT-74325 это будет как отсутствие ограничения.
                    type = NetworkViewType.not_limited;
                    percent = 0;
                    break;
                default:
                    type = NetworkViewType.percent;
                    percent = contextLimit;
            }
            return new NetworkView(type, percent, contextPriceCoef, enableCpcHold);
        }

        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> forward) {
            BoolTransition enableCpcHold = owner.getOptsChanges().get(CampaignOpts.ENABLE_CPC_HOLD);
            NetworkView before;
            // DIRECT-82048 Если поле platform не записано в строке таблицы, то берётся такой platform,
            // будто показы в сетях включены.
            String platformBefore = ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform.both.getLiteral();
            if (owner.getRecord().getOperation() == Operation.INSERT) {
                before = DISABLED;
            } else {
                String contextLimitRaw;
                String contextPriceCoefRaw;
                try {
                    contextLimitRaw = Objects.requireNonNull(
                            owner.getBeforeFields().get(CAMPAIGNS.CONTEXT_LIMIT.getName()));
                    contextPriceCoefRaw = Objects.requireNonNull(
                            owner.getBeforeFields().get(CAMPAIGNS.CONTEXT_PRICE_COEF.getName()));
                } catch (NullPointerException ignored) {
                    // DIRECT-75353
                    // Эту обёртку над NPE стоит убрать, когда будет продакшновая БД.
                    // Из-за баги, исправленной в DIRECT-75417, часть событий о смене настроек сетей
                    // в тестовой БД содержит не все необходимые данные.
                    return;
                }
                try {
                    platformBefore = Objects.requireNonNull(owner.getBeforeFields().get(CAMPAIGNS.PLATFORM.getName()));
                } catch (NullPointerException ignored) {
                    // DIRECT-82048 Желание смотреть в поле platform появилось через пару месяцев после того,
                    // как пользологи выехали в продакшн. В базе существуют записи, в которых указано изменение
                    // ContextLimit/ContextPriceCoef, но не указано platform. Для таких записей считается как и до этого
                    // изменения: в сетях показывать.
                }
                before = makeNetworkView(
                        platformBefore,
                        Integer.parseInt(contextLimitRaw),
                        Integer.parseInt(contextPriceCoefRaw),
                        enableCpcHold.oldValue);
            }
            String platformAfter;
            try {
                platformAfter = Objects.requireNonNull(owner.getAfterFields().get(CAMPAIGNS.PLATFORM.getName()));
            } catch (NullPointerException ignored) {
                // DIRECT-82048 см. комментарии выше.
                platformAfter = platformBefore;
            }

            NetworkView after = makeNetworkView(
                    platformAfter,
                    Integer.parseInt(Objects.requireNonNull(
                            owner.getFromAfterOrBefore(CAMPAIGNS.CONTEXT_LIMIT.getName()))),
                    Integer.parseInt(Objects.requireNonNull(
                            owner.getFromAfterOrBefore(CAMPAIGNS.CONTEXT_PRICE_COEF.getName()))),
                    enableCpcHold.newValue);
            if (!before.equals(after)
                    // Отключение настроек в сетях возможно только при изменении параметров стратегии на такие,
                    // которые запрещают показы в сетях. В этом случае об этом будет сказано в событии изменения
                    // стратегии.
                    && !after.equals(DISABLED)) {
                owner.addEvent(new CampaignNetworkEvent(owner.getClientId(), owner.getCampaignId(),
                        before, after));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_NETWORK);
        }
    }

    private static class OptsCaseHandler implements CaseHandler {
        private final NetworkCaseHandler networkCaseHandler;

        private OptsCaseHandler(NetworkCaseHandler networkCaseHandler) {
            this.networkCaseHandler = networkCaseHandler;
        }

        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> forward) {

            var opts = owner.getOptsChanges();
            var clientId = owner.getClientId();
            var campaignId = owner.getCampaignId();

            var noExtendedGeoTargeting = opts.get(CampaignOpts.NO_EXTENDED_GEOTARGETING);
            if (noExtendedGeoTargeting.isChange) {
                owner.addEvent(new CampaignValueChangeEvent(
                        clientId,
                        campaignId,
                        OutputCategory.CAMPAIGN_EXTENDED_GEOTARGETING,
                        // В интерфейсе показываем ON если флаг включен.
                        // В базе есть опция, если флаг выключен.
                        // Поэтому инвертируем показания.
                        Boolean.toString(noExtendedGeoTargeting.newValue),
                        Boolean.toString(noExtendedGeoTargeting.oldValue)));
            }

            var enableCPCHold = opts.get(CampaignOpts.ENABLE_CPC_HOLD);
            if (enableCPCHold.isChange) {
                forward.accept(networkCaseHandler);
            }

            var recommendationsManagementEnabled = opts.get(CampaignOpts.RECOMMENDATIONS_MANAGEMENT_ENABLED);
            var priceRecommendationsManagementEnabled = opts.get(CampaignOpts.PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED);

            var autoApplyPermissionsOld = new HashSet<String>();
            if (recommendationsManagementEnabled.oldValue) {
                autoApplyPermissionsOld.add(CampaignOpts.RECOMMENDATIONS_MANAGEMENT_ENABLED.toString());
            }
            if (priceRecommendationsManagementEnabled.oldValue) {
                autoApplyPermissionsOld.add(CampaignOpts.PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED.toString());
            }

            var autoApplyPermissionsNew = new HashSet<String>();
            if (recommendationsManagementEnabled.newValue) {
                autoApplyPermissionsNew.add(CampaignOpts.RECOMMENDATIONS_MANAGEMENT_ENABLED.toString());
            }
            if (priceRecommendationsManagementEnabled.newValue) {
                autoApplyPermissionsNew.add(CampaignOpts.PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED.toString());
            }

            if (!autoApplyPermissionsOld.equals(autoApplyPermissionsNew)) {
                owner.addEvent(new CampaignListChangeEvent(
                        clientId,
                        campaignId,
                        OutputCategory.CAMPAIGN_AUTO_APPLY_PERMISSIONS,
                        autoApplyPermissionsOld,
                        autoApplyPermissionsNew
                ));
            }

        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_EXTENDED_GEOTARGETING,
                    OutputCategory.CAMPAIGN_AUTO_APPLY_PERMISSIONS);
        }
    }

    private static class RegionsCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> forward) {
            final String name = CAMPAIGNS.GEO.getName();
            List<Integer> oldRegions = splitCommaDelimited(owner.getBeforeFields().get(name))
                    .map(Integer::parseInt)
                    .collect(Collectors.toList());
            List<Integer> newRegions = splitCommaDelimited(owner.getAfterFields().get(name))
                    .map(Integer::parseInt)
                    .collect(Collectors.toList());
            if (!oldRegions.equals(newRegions)) {
                owner.addEvent(new CampaignRegionsEvent(owner.getClientId(), owner.getCampaignId(),
                        oldRegions, newRegions));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_REGIONS);
        }
    }

    private static class RestrictFrequencyCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            // У обоих полей значение по умолчанию - 0
            RestrictFrequencyView oldView =
                    new RestrictFrequencyView(0, 0);
            if (owner.getRecord().getOperation() == Operation.UPDATE) {
                try {
                    oldView = new RestrictFrequencyView(
                            Integer.parseInt(Objects.requireNonNull(
                                    owner.getBeforeFields().get(CAMPAIGNS.RF.getName()))),
                            Integer.parseInt(Objects.requireNonNull(
                                    owner.getBeforeFields().get(CAMPAIGNS.RF_RESET.getName()))));
                } catch (NullPointerException ignored) {
                    // Это событие появилось не сразу, поэтому для него может не быть данных.
                    return;
                }
            }
            RestrictFrequencyView
                    newView = new RestrictFrequencyView(
                    Integer.parseInt(Objects.requireNonNull(
                            owner.getFromAfterOrBefore(CAMPAIGNS.RF.getName()))),
                    Integer.parseInt(Objects.requireNonNull(
                            owner.getFromAfterOrBefore(CAMPAIGNS.RF_RESET.getName()))));
            if (!Objects.equals(oldView, newView)) {
                owner.addEvent(new CampaignRestrictFrequencyEvent(owner.getClientId(),
                        owner.getCampaignId(), oldView, newView));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_RESTRICT_FREQUENCY);
        }
    }

    private static class StartTimeCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> forward) {
            String name = CAMPAIGNS.START_TIME.getName();
            LocalDate oldStartDate =
                    Optional.ofNullable(owner.getBeforeFields().get(name)).map(LocalDate::parse).orElse(null);
            LocalDate newStartDate = LocalDate.parse(owner.getAfterFields().get(name));
            if (!Objects.equals(oldStartDate, newStartDate)) {
                owner.addEvent(new CampaignValueChangeEvent(owner.getClientId(),
                        owner.getCampaignId(),
                        OutputCategory.CAMPAIGN_START_TIME,
                        Optional.ofNullable(oldStartDate).map(LocalDate::toString).orElse(null),
                        newStartDate.toString()));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_START_TIME);
        }
    }

    private static class StatusShowCaseHandler implements CaseHandler {
        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            BoolTransition transition = owner.yesNoTransition(CAMPAIGNS.STATUS_SHOW.getName());
            if (transition != null && transition.isChange) {
                owner.addEvent(new CampaignStatusChangeEvent(owner.getClientId(),
                        owner.getCampaignId(),
                        transition.newValue ? OutputCategory.CAMPAIGN_SHOW : OutputCategory.CAMPAIGN_HIDE));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_HIDE, OutputCategory.CAMPAIGN_SHOW);
        }
    }

    private static class StrategyCaseHandler implements CaseHandler {
        private static final GdStrategyExtractorFacade gdStrategyExtractorFacade =
                new GdStrategyExtractorFacade(STRATEGIES_EXTRACTORS_BY_TYPES);


        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> visitAnotherOne) {
            CurrencyCode currencyCode;
            try {
                // Валюта у кампании не меняется. Из after будет взято только при operation=INSERT
                currencyCode = CurrencyCode.valueOf(Objects.requireNonNull(
                        owner.getFromAfterOrBefore(CAMPAIGNS.CURRENCY.getName())));
            } catch (NullPointerException ignored) {
                // См. аналогичный комментарий в makeStrategyView
                return;
            }
            GdCampaignFlatStrategy oldFlatStrategy = makeFlatStrategyView(owner.getBeforeFields()::get);
            GdCampaignFlatStrategy newFlatStrategy = makeFlatStrategyView(owner::getFromAfterOrBefore);
            GdCampaignFlatStrategy oldStrategy = makeStrategyView(owner.getBeforeFields()::get);
            GdCampaignFlatStrategy newStrategy = makeStrategyView(owner::getFromAfterOrBefore);

            if ((oldFlatStrategy != null || owner.getRecord().getOperation() == Operation.INSERT)
                    && ((newFlatStrategy != null && !Objects.equals(oldFlatStrategy, newFlatStrategy))
                    || (newStrategy != null && !Objects.equals(oldStrategy, newStrategy)))) {
                owner.addEvent(new CampaignStrategyEvent(owner.getClientId(), owner.getCampaignId(),
                        currencyCode, oldFlatStrategy, newFlatStrategy, oldStrategy, newStrategy));
            }
        }


        private GdCampaignFlatStrategy makeFlatStrategyView(Function<String, String> fieldGetter) {
            GdiCampaignStrategyGroup strategyGroup = getStrategyGroup(fieldGetter);
            if (strategyGroup == null) {
                return null;
            }
            return gdStrategyExtractorFacade.extractFlatStrategy(strategyGroup);
        }

        private GdCampaignFlatStrategy makeStrategyView(Function<String, String> fieldGetter) {
            GdiCampaignStrategyGroup strategyGroup = getStrategyGroup(fieldGetter);
            if (strategyGroup == null) {
                return null;
            }
            return gdStrategyExtractorFacade.extractStrategy(strategyGroup);
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_STRATEGY);
        }

    }

    private static GdiCampaignStrategyGroup getStrategyGroup(Function<String, String> fieldGetter) {
        StrategyGroupImpl strategyGroup = new StrategyGroupImpl()
                .withOptsStrategyName(null); //TODO DIRECT-78569
        try {
            strategyGroup.setDayBudget(new BigDecimal(Objects.requireNonNull(
                    fieldGetter.apply(CAMPAIGNS.DAY_BUDGET.getName()))));
            strategyGroup.setDayBudgetShowMode(GdiDayBudgetShowMode.fromSource(
                    Convert.convert(
                            Objects.requireNonNull(
                                    fieldGetter.apply(CAMPAIGNS.DAY_BUDGET_SHOW_MODE.getName())),
                            CampaignsDayBudgetShowMode.class)));
            strategyGroup.setPlatform(CampaignsPlatform.fromSource(
                    Convert.convert(
                            Objects.requireNonNull(fieldGetter.apply(CAMPAIGNS.PLATFORM.getName())),
                            ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform.class)));
            strategyGroup.setStrategyData(Objects.requireNonNull(
                    fieldGetter.apply(CAMPAIGNS.STRATEGY_DATA.getName())));
            strategyGroup.setStrategyName(GdiCampaignStrategyName.fromSource(
                    Convert.convert(
                            Objects.requireNonNull(fieldGetter.apply(CAMPAIGNS.STRATEGY_NAME.getName())),
                            CampaignsStrategyName.class)));
            strategyGroup.setType(CampaignType.fromSource(
                    Convert.convert(
                            Objects.requireNonNull(fieldGetter.apply(CAMPAIGNS.TYPE.getName())),
                            ru.yandex.direct.dbschema.ppc.enums.CampaignsType.class)));
        } catch (NullPointerException ignored) {
            // В тестовой базе до сих пор есть строки, в которых присутствуют не все необходимые поля.
            // Вместо того, чтобы падать на таких строках с ошибкой, читалка будет их игнорировать.
            // После того, как наколенный кликхаус из докерных контейнеров на ppcdev6 исчезнет,
            // можно (но не обязательно нужно) будет удалить этот catch-блок.
            return null;
        }
        if (strategyGroup.getStrategyName() == GdiCampaignStrategyName.AUTOBUDGET_MEDIA) {
            // Стратегия из БаЯна, с единственной возможной настройкой “стараться распределять до даты Х”.
            // Баян закрывается, решили такие стратегии не поддерживать.
            return null;
        }
        return strategyGroup;
    }

    private static class TimeTargetCaseHandler implements CaseHandler {
        private static final HoursCoef HOURS_COEF_ALL_100;
        private static final TimeTarget TIME_TARGET_ALL_100;

        static {
            HOURS_COEF_ALL_100 = new HoursCoef();
            for (int hour = 0; hour < 24; ++hour) {
                HOURS_COEF_ALL_100.setCoef(hour, 100);
            }
            TIME_TARGET_ALL_100 = new TimeTarget();
            for (WeekdayType weekdayType : WeekdayType.values()) {
                TIME_TARGET_ALL_100.setWeekdayCoef(weekdayType, HOURS_COEF_ALL_100);
            }
        }

        private TimeTargetView parseRawString(long timeZone, @Nullable String timeTargetRaw) {
            Map<WeekdayType, List<Integer>> arrays =
                    ImmutableMap.<WeekdayType, List<Integer>>builder()
                            .put(WeekdayType.SUNDAY, new ArrayList<>(24))
                            .put(WeekdayType.MONDAY, new ArrayList<>(24))
                            .put(WeekdayType.TUESDAY, new ArrayList<>(24))
                            .put(WeekdayType.WEDNESDAY, new ArrayList<>(24))
                            .put(WeekdayType.THURSDAY, new ArrayList<>(24))
                            .put(WeekdayType.FRIDAY, new ArrayList<>(24))
                            .put(WeekdayType.SATURDAY, new ArrayList<>(24))
                            .put(WeekdayType.HOLIDAY, new ArrayList<>(24))
                            .put(WeekdayType.WORKING_WEEKEND, new ArrayList<>(24))
                            .build();

            // Когда таймтаргетинг равен null или пустой строке, все коэффициенты в нём 0 или 100 - это всё идентичные
            // значения, говорящие о том, что включены все дни и все часы без корректировок по времени.
            Map<WeekdayType, HoursCoef> timeTargetMap = TIME_TARGET_ALL_100.getWeekdayCoefs();
            if (!StringUtils.isEmpty(timeTargetRaw)) {
                timeTargetMap = TimeTarget.parseRawString(timeTargetRaw).getWeekdayCoefs();
                boolean allZeros = true;
                Iterator<HoursCoef> hoursCoefIterator = timeTargetMap.values().iterator();
                while (allZeros && hoursCoefIterator.hasNext()) {
                    HoursCoef hoursCoef = hoursCoefIterator.next();
                    for (int hour = 0; allZeros && hour < 24; ++hour) {
                        allZeros = hoursCoef.getCoefForHour(hour) == 0;
                    }
                }
                if (allZeros) {
                    timeTargetMap = TIME_TARGET_ALL_100.getWeekdayCoefs();
                }
            }

            for (Map.Entry<WeekdayType, List<Integer>> pair : arrays.entrySet()) {
                HoursCoef hoursCoef = timeTargetMap.getOrDefault(pair.getKey(), HOURS_COEF_ALL_100);
                for (int hour = 0; hour < 24; ++hour) {
                    pair.getValue().add(hoursCoef.getCoefForHour(hour));
                }
            }
            return new TimeTargetView(timeZone, arrays);
        }

        @Override
        public void visitOnce(CampaignSingleRecordMetaData owner,
                              Consumer<RunOnce.Visitor<CampaignSingleRecordMetaData>> forward) {
            TimeTargetView oldTimeTargetView = parseRawString(
                    Optional.ofNullable(owner.getBeforeFields().get(CAMPAIGNS.TIMEZONE_ID.getName()))
                            .map(Long::parseLong).orElse(0L),
                    owner.getBeforeFields().get(CAMPAIGNS.TIME_TARGET.getName()));
            TimeTargetView newTimeTargetView = parseRawString(
                    Optional.ofNullable(owner.getFromAfterOrBefore(CAMPAIGNS.TIMEZONE_ID.getName()))
                            .map(Long::parseLong).orElse(0L),
                    owner.getFromAfterOrBefore(CAMPAIGNS.TIME_TARGET.getName()));
            if (!oldTimeTargetView.equals(newTimeTargetView)) {
                owner.addEvent(new CampaignTimeTargetEvent(owner.getClientId(), owner.getCampaignId(),
                        oldTimeTargetView, newTimeTargetView));
            }
        }

        @Override
        public ImmutableSet<OutputCategory> getCategories() {
            return ImmutableSet.of(OutputCategory.CAMPAIGN_TIME_TARGET);
        }
    }
}
