package ru.yandex.direct.useractionlog.reader.generator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.useractionlog.schema.ActionLogSchema;
import ru.yandex.direct.useractionlog.schema.Operation;

@ParametersAreNonnullByDefault
public class FieldKeyValueUtil {
    private static final String HAS_FIELD_QUERY_TEMPLATE = String.format(
            "arrayExists(f -> (f in (%%1$s)), if(%1$s = '%2$s',  %3$s, %4$s))",
            ActionLogSchema.OPERATION.getExpr(),
            Operation.INSERT.name(),
            ActionLogSchema.NEW_FIELDS_NAMES.getExpr(),
            ActionLogSchema.OLD_FIELDS_NAMES.getExpr());
    private static final String CHECK_VALUE_QUERY_TEMPLATE = String.format(
            "(%1$s IN ('%2$s', '%3$s') AND has(%5$s, ?) AND arrayElement(%6$s, indexOf(%5$s, ?)) IN (%%1$s) "
                    + "OR %1$s = '%4$s' AND has(%7$s, ?) and arrayElement(%8$s, indexOf(%7$s, ?)) IN (%%1$s))",
            ActionLogSchema.OPERATION.getExpr(),
            Operation.UPDATE.name(),
            Operation.DELETE.name(),
            Operation.INSERT.name(),
            ActionLogSchema.OLD_FIELDS_NAMES.getExpr(),
            ActionLogSchema.OLD_FIELDS_VALUES.getExpr(),
            ActionLogSchema.NEW_FIELDS_NAMES.getExpr(),
            ActionLogSchema.NEW_FIELDS_VALUES.getExpr());


    private FieldKeyValueUtil() {
    }

    /**
     * Трансформация в sql условие WHERE для получения из таблицы пользологов только тех
     * строк, в которых произошли нужные изменения. Поля без значений группируются отдельно
     * и для них проверяется, что либо в old_fields.name, либо в new_fields.name есть
     * хотя бы одно из них. Поля со значением группируются по ключу и для них проверяется,
     * что соответствующее значение поля является одним из заданных.
     *
     * @param values набор полей со значениями для фильтрации
     * @return условие WHERE и набор привязок для него
     */
    public static Pair<String, List<String>> toSqlQuery(Collection<FieldKeyValue> values) {
        Set<FieldKeyValue> withoutValue = new HashSet<>();
        Map<String, Set<FieldKeyValue>> withValueMap = new HashMap<>();

        for (FieldKeyValue item : values) {
            if (item.getValue().isPresent()) {
                withValueMap.putIfAbsent(item.getKey(), new HashSet<>());
                withValueMap.get(item.getKey()).add(item);
            } else {
                withoutValue.add(item);
            }
        }

        List<String> queries = new ArrayList<>();
        List<String> bindings = new ArrayList<>();
        if (!withoutValue.isEmpty()) {
            queries.add(String.format(
                    HAS_FIELD_QUERY_TEMPLATE, StreamEx.of(withoutValue).map(x -> "?").joining(", ")));
            for (FieldKeyValue item : withoutValue) {
                bindings.add(item.getKey());
            }
        }

        for (Map.Entry<String, Set<FieldKeyValue>> entry : withValueMap.entrySet()) {
            queries.add(String.format(
                    CHECK_VALUE_QUERY_TEMPLATE, StreamEx.of(entry.getValue()).map(x -> "?").joining(", ")));
            for (int index = 0; index < 2; index++) {
                bindings.add(entry.getKey());
                bindings.add(entry.getKey());
                for (FieldKeyValue keyValue : entry.getValue()) {
                    bindings.add(keyValue.getValue().get());
                }
            }
        }
        return Pair.of("(" + String.join(" OR ", queries) + ")", bindings);
    }
}
