package ru.yandex.direct.useractionlog.reader.generator;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.PeekingIterator;

import ru.yandex.direct.useractionlog.db.ReadActionLogTable;
import ru.yandex.direct.useractionlog.reader.model.LogRecord;
import ru.yandex.direct.useractionlog.reader.model.OutputCategory;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;

@ParametersAreNonnullByDefault
public class LogRecordGeneratorByType implements LogRecordGenerator {
    private final ImmutableMap<String, LogRecordGenerator> generators;

    public LogRecordGeneratorByType(ReadActionLogTable.Order order) {
        List<LogRecordGenerator> generatorList = Arrays.asList(
                new AdGroupLogRecordGenerator(order),
                new CampaignsLogRecordGenerator(order),
                new MultipliersLogRecordGenerator(),
                new BannersLogRecordGenerator(order));
        ImmutableMap.Builder<String, LogRecordGenerator> builder = ImmutableMap.builder();
        for (LogRecordGenerator generator : generatorList) {
            for (String type : generator.getSupportedActionLogRecordTypeToFields().keySet()) {
                builder.put(type, generator);
            }
        }
        this.generators = builder.build();
    }

    @Override
    public List<LogRecord> offer(PeekingIterator<ActionLogRecordWithStats> recordIter) {
        LogRecordGenerator generator = generators.get(recordIter.peek().getRecord().getType());
        if (generator != null) {
            return generator.offer(recordIter);
        } else {
            return Collections.emptyList();
        }
    }

    @Override
    public Map<String, Collection<FieldKeyValue>> getSupportedActionLogRecordTypeToFields() {
        Map<String, Collection<FieldKeyValue>> result = new HashMap<>();
        for (LogRecordGenerator generator : generators.values()) {
            for (Map.Entry<String, Collection<FieldKeyValue>> entry :
                    generator.getSupportedActionLogRecordTypeToFields().entrySet()) {
                result.putIfAbsent(entry.getKey(), new HashSet<>());
                result.get(entry.getKey()).addAll(entry.getValue());
            }
        }
        return result;
    }

    @Override
    public Map<String, Collection<FieldKeyValue>> getActionLogRecordTypeToFields(
            Collection<OutputCategory> categories) {
        Map<String, Collection<FieldKeyValue>> result = new HashMap<>();
        for (LogRecordGenerator generator : generators.values()) {
            for (Map.Entry<String, Collection<FieldKeyValue>> entry :
                    generator.getActionLogRecordTypeToFields(categories).entrySet()) {
                result.putIfAbsent(entry.getKey(), new HashSet<>());
                result.get(entry.getKey()).addAll(entry.getValue());
            }
        }
        return result;
    }
}
