package ru.yandex.direct.useractionlog.reader.generator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.google.common.collect.PeekingIterator;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbschema.ppc.enums.DemographyMultiplierValuesAge;
import ru.yandex.direct.dbschema.ppc.enums.DemographyMultiplierValuesGender;
import ru.yandex.direct.dbschema.ppc.enums.HierarchicalMultipliersType;
import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.ClientId;
import ru.yandex.direct.useractionlog.model.HierarchicalMultipliersData;
import ru.yandex.direct.useractionlog.reader.model.DemographyMultipliersEvent;
import ru.yandex.direct.useractionlog.reader.model.InputCategory;
import ru.yandex.direct.useractionlog.reader.model.LogEvent;
import ru.yandex.direct.useractionlog.reader.model.LogRecord;
import ru.yandex.direct.useractionlog.reader.model.OutputCategory;
import ru.yandex.direct.useractionlog.reader.model.RetargetingMultipliersEvent;
import ru.yandex.direct.useractionlog.reader.model.SingleMultiplierEvent;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogRecordWithStats;
import ru.yandex.direct.useractionlog.schema.ObjectPath;
import ru.yandex.direct.useractionlog.schema.Operation;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

@ParametersAreNonnullByDefault
public class MultipliersLogRecordGenerator extends DefaultRecordGenerator {
    /**
     * Время от времени добавляются новые типы корректировок. Они добавляются в UserActionLogWriter,
     * но фронтенд может не успеть реализовать на своей стороне поддержку новых корректировок.
     * Поэтому все неожиданные для фронтенда корректировки пропускаются.
     */
    private static final ImmutableSet<String> SUPPORTED_TYPES = ImmutableSet.of(
            HierarchicalMultipliersType.mobile_multiplier.getLiteral(),
            HierarchicalMultipliersType.demography_multiplier.getLiteral(),
            HierarchicalMultipliersType.performance_tgo_multiplier.getLiteral(),
            HierarchicalMultipliersType.retargeting_multiplier.getLiteral(),
            HierarchicalMultipliersType.distance_multiplier.getLiteral(),
            HierarchicalMultipliersType.video_multiplier.getLiteral());

    /**
     * Галка "Выключить корректировки" бывает только на сложносоставных корректировках. Выключение корректировок
     * показывается как удаление всех корректировок. Включение - как добавление всех корректировок. DIRECT-77628
     */
    private static final ImmutableSet<String> TOGGLEABLE_TYPES = ImmutableSet.of(
            HierarchicalMultipliersType.demography_multiplier.getLiteral(),
            HierarchicalMultipliersType.retargeting_multiplier.getLiteral());

    private static final Logger logger = LoggerFactory.getLogger(MultipliersLogRecordGenerator.class);
    private static final Map<OutputCategory, String> TABLE_BY_CATEGORY = Maps.asMap(
            InputCategory.HIERARCHICAL_MULTIPLIERS.toOutputCategories(),
            ignored -> PPC.HIERARCHICAL_MULTIPLIERS.getName());

    private static final Map<OutputCategory, Collection<FieldKeyValue>> FIELDS_VALUES_BY_CATEGORY =
            Maps.transformValues(
                    ImmutableMap.<OutputCategory, HierarchicalMultipliersType>builder()
                            .put(OutputCategory.ADGROUP_DEMOGRAPHY_MULTIPLIERS,
                                    HierarchicalMultipliersType.demography_multiplier)
                            .put(OutputCategory.ADGROUP_MOBILE_MULTIPLIER,
                                    HierarchicalMultipliersType.mobile_multiplier)
                            .put(OutputCategory.ADGROUP_PERFORMANCE_TGO_MULTIPLIER,
                                    HierarchicalMultipliersType.performance_tgo_multiplier)
                            .put(OutputCategory.ADGROUP_RETARGETING_MULTIPLIERS,
                                    HierarchicalMultipliersType.retargeting_multiplier)
                            .put(OutputCategory.ADGROUP_VIDEO_MULTIPLIER,
                                    HierarchicalMultipliersType.video_multiplier)
                            .put(OutputCategory.CAMPAIGN_DEMOGRAPHY_MULTIPLIERS,
                                    HierarchicalMultipliersType.demography_multiplier)
                            .put(OutputCategory.CAMPAIGN_MOBILE_MULTIPLIER,
                                    HierarchicalMultipliersType.mobile_multiplier)
                            .put(OutputCategory.CAMPAIGN_PERFORMANCE_TGO_MULTIPLIER,
                                    HierarchicalMultipliersType.performance_tgo_multiplier)
                            .put(OutputCategory.CAMPAIGN_RETARGETING_MULTIPLIERS,
                                    HierarchicalMultipliersType.retargeting_multiplier)
                            .put(OutputCategory.CAMPAIGN_VIDEO_MULTIPLIER,
                                    HierarchicalMultipliersType.video_multiplier)
                            .build(),
                    v -> Collections.singletonList(FieldKeyValue.of(
                            PPC.HIERARCHICAL_MULTIPLIERS.TYPE.getName(),
                            Optional.of(Objects.requireNonNull(v).getLiteral()))));

    static {
        if (!TABLE_BY_CATEGORY.keySet().equals(FIELDS_VALUES_BY_CATEGORY.keySet())) {
            throw new IllegalStateException("mismatch between specified categories");
        }
    }

    private static Optional<HierarchicalMultipliersData.Root> deserializeBefore(ActionLogRecord record) {
        HierarchicalMultipliersData.Root result = new HierarchicalMultipliersData.Root();
        Map<String, String> firstRecordOldMap = record.getOldFields().toMap();
        if (record.getOperation() == Operation.INSERT) {
            return Optional.empty();
        } else {
            return Optional.of(result.updateFromMap(firstRecordOldMap));
        }
    }

    private static Optional<HierarchicalMultipliersData.Root> deserializeAfter(ActionLogRecord record) {
        HierarchicalMultipliersData.Root result = new HierarchicalMultipliersData.Root();
        Map<String, String> firstRecordNewMap = record.getNewFields().toMap();
        if (record.getOperation() == Operation.INSERT) {
            return Optional.of(result.updateFromMap(firstRecordNewMap));
        } else if (record.getOperation() == Operation.UPDATE) {
            return Optional.of(result.updateFromMaps(record.getOldFields().toMap(), firstRecordNewMap));
        } else {
            return Optional.empty();
        }
    }

    /**
     * Берёт из итератора группу записей. В этой группе все записи:
     * <ul>
     * <li>Отсортированы по возрастанию времени.</li>
     * <li>Связаны с корректировками ставок одного и того же типа.</li>
     * <li>Связаны с корректировками ставок для одной и той же кампании или группы баннеров.</li>
     * <li>Имеют один и тот же {@link ActionLogRecord#getGtid()}</li>
     * </ul>
     */
    private static List<ActionLogRecordWithStats> nextRecordGroup(
            PeekingIterator<ActionLogRecordWithStats> recordIter) {
        List<ActionLogRecordWithStats> result = new ArrayList<>();
        final ActionLogRecordWithStats firstRecordWithStats = recordIter.next();
        result.add(firstRecordWithStats);
        String type = deserializeBefore(firstRecordWithStats.getRecord())
                .orElseGet(() -> deserializeAfter(firstRecordWithStats.getRecord())
                        .orElseThrow(() -> new IllegalStateException("Can't get type")))
                .getType();
        if (!SUPPORTED_TYPES.contains(type)) {
            return result;
        }
        while (recordIter.hasNext()) {
            ActionLogRecordWithStats candidate = recordIter.peek();
            if (!candidate.getRecord().getType().equals(firstRecordWithStats.getRecord().getType())
                    || !candidate.getRecord().getGtid().equals(firstRecordWithStats.getRecord().getGtid())
                    || !candidate.getRecord().getPath().equals(firstRecordWithStats.getRecord().getPath())) {
                break;
            }
            String newType = deserializeBefore(candidate.getRecord())
                    .orElseGet(() -> deserializeAfter(candidate.getRecord())
                            .orElseThrow(() -> new IllegalStateException("Can't get type")))
                    .getType();
            if (!type.equals(newType)) {
                break;
            }
            result.add(recordIter.next());
        }
        // Так как все записи в списке имеют один и тот же gtid, то достаточно отсортировать
        // по querySerial и rowSerial
        result.sort(Comparator
                .comparingInt(r -> ((ActionLogRecordWithStats) r).getRecord().getQuerySerial())
                .thenComparingInt(r -> ((ActionLogRecordWithStats) r).getRecord().getRowSerial()));
        return result;
    }

    @Nullable
    private static LogEvent demographyMultipliersEvent(ObjectPath objectPath,
                                                       HierarchicalMultipliersData.Root oldData,
                                                       HierarchicalMultipliersData.Root newData,
                                                       ActionLogRecord firstRecord) {
        return makeMultipliersEvent(
                objectPath,
                oldData,
                newData,
                data -> data.getAllRelatedDatabaseIds().stream()
                        .map(id -> (HierarchicalMultipliersData.Demography) data.getRelated(id))
                        .map(demo -> asDemographyMultiplierView(demo, firstRecord))
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList()),
                DemographyMultipliersEvent::new);
    }

    @Nullable
    private static DemographyMultipliersEvent.DemographyMultiplierView asDemographyMultiplierView(
            HierarchicalMultipliersData.Demography demography, ActionLogRecord firstRecord) {
        DemographyMultipliersEvent.Age age;
        if (StringUtils.isEmpty(demography.getAge())) {
            age = DemographyMultipliersEvent.Age.any;
        } else if (demography.getAge().equals(DemographyMultiplierValuesAge._0_17.getLiteral())) {
            age = DemographyMultipliersEvent.Age.age_0_17;
        } else if (demography.getAge().equals(DemographyMultiplierValuesAge._18_24.getLiteral())) {
            age = DemographyMultipliersEvent.Age.age_18_24;
        } else if (demography.getAge().equals(DemographyMultiplierValuesAge._25_34.getLiteral())) {
            age = DemographyMultipliersEvent.Age.age_25_34;
        } else if (demography.getAge().equals(DemographyMultiplierValuesAge._35_44.getLiteral())) {
            age = DemographyMultipliersEvent.Age.age_35_44;
        } else if (demography.getAge().equals(DemographyMultiplierValuesAge._45_.getLiteral())) {
            age = DemographyMultipliersEvent.Age.age_45_inf;  // DIRECT-77321
        } else if (demography.getAge().equals(DemographyMultiplierValuesAge._45_54.getLiteral())) {
            age = DemographyMultipliersEvent.Age.age_45_54;
        } else if (demography.getAge().equals(DemographyMultiplierValuesAge._55_.getLiteral())) {
            age = DemographyMultipliersEvent.Age.age_55_inf;
        } else {
            logger.warn("Unknown age {}. First of merged records: gtid={} qs={} rs={}",
                    demography.getAge(), firstRecord.getGtid(), firstRecord.getQuerySerial(),
                    firstRecord.getRowSerial());
            return null;
        }
        DemographyMultipliersEvent.Gender gender;
        if (StringUtils.isEmpty(demography.getGender())) {
            gender = DemographyMultipliersEvent.Gender.any;
        } else if (demography.getGender().equals(DemographyMultiplierValuesGender.female.getLiteral())) {
            gender = DemographyMultipliersEvent.Gender.female;
        } else if (demography.getGender().equals(DemographyMultiplierValuesGender.male.getLiteral())) {
            gender = DemographyMultipliersEvent.Gender.male;
        } else {
            logger.warn("Unknown gender {}. First of merged records: gtid={} qs={} rs={}",
                    demography.getGender(), firstRecord.getGtid(), firstRecord.getQuerySerial(),
                    firstRecord.getRowSerial());
            return null;
        }
        return new DemographyMultipliersEvent.DemographyMultiplierView(age, gender,
                Integer.parseInt(demography.getMultiplierPct()));
    }

    @Nullable
    private static LogEvent mobileMultiplierEvent(ObjectPath objectPath,
                                                  HierarchicalMultipliersData.Root oldData,
                                                  HierarchicalMultipliersData.Root newData) {
        return makeMultipliersEvent(
                objectPath,
                oldData,
                newData,
                MultipliersLogRecordGenerator::parseMultiplier,
                singleMultiplierEventFactory(
                        OutputCategory.CAMPAIGN_MOBILE_MULTIPLIER,
                        OutputCategory.ADGROUP_MOBILE_MULTIPLIER));
    }

    @Nullable
    private static LogEvent performanceTgoMultiplierEvent(ObjectPath objectPath,
                                                          HierarchicalMultipliersData.Root oldData,
                                                          HierarchicalMultipliersData.Root newData) {
        return makeMultipliersEvent(
                objectPath,
                oldData,
                newData,
                MultipliersLogRecordGenerator::parseMultiplier,
                singleMultiplierEventFactory(
                        OutputCategory.CAMPAIGN_PERFORMANCE_TGO_MULTIPLIER,
                        OutputCategory.ADGROUP_PERFORMANCE_TGO_MULTIPLIER));
    }

    @Nullable
    private static LogEvent retargetingMultipliersEvent(ObjectPath objectPath,
                                                        HierarchicalMultipliersData.Root oldData,
                                                        HierarchicalMultipliersData.Root newData) {
        return makeMultipliersEvent(
                objectPath,
                oldData,
                newData,
                data -> data.getAllRelatedDatabaseIds().stream()
                        .map(id -> (HierarchicalMultipliersData.Retargeting) data.getRelated(id))
                        .map(retargeting -> new RetargetingMultipliersEvent.RetargetingMultiplierView(
                                Long.parseLong(retargeting.getRetCondId()),
                                retargeting.getRetCondName(),
                                Integer.parseInt(retargeting.getMultiplierPct())))
                        .collect(Collectors.toList()),
                RetargetingMultipliersEvent::new);
    }

    @Nullable
    private static LogEvent videoMultiplierEvent(ObjectPath objectPath,
                                                 HierarchicalMultipliersData.Root oldData,
                                                 HierarchicalMultipliersData.Root newData) {
        return makeMultipliersEvent(
                objectPath,
                oldData,
                newData,
                MultipliersLogRecordGenerator::parseMultiplier,
                singleMultiplierEventFactory(
                        OutputCategory.ADGROUP_VIDEO_MULTIPLIER,
                        OutputCategory.CAMPAIGN_VIDEO_MULTIPLIER));
    }

    @Nullable
    private static <V> LogEvent makeMultipliersEvent(ObjectPath objectPath,
                                                     HierarchicalMultipliersData.Root oldData,
                                                     HierarchicalMultipliersData.Root newData,
                                                     Function<HierarchicalMultipliersData.Root, V> viewFactory,
                                                     MultiplierEventFactory<V> eventFactory) {
        ClientId clientId;
        CampaignId campaignId;
        AdGroupId adGroupId;
        if (objectPath instanceof ObjectPath.CampaignPath) {
            clientId = ((ObjectPath.CampaignPath) objectPath).getClientId();
            campaignId = (CampaignId) objectPath.getId();
            adGroupId = null;
        } else if (objectPath instanceof ObjectPath.AdGroupPath) {
            clientId = ((ObjectPath.AdGroupPath) objectPath).getClientId();
            campaignId = ((ObjectPath.AdGroupPath) objectPath).getCampaignId();
            adGroupId = (AdGroupId) objectPath.getId();
        } else {
            throw new UnsupportedOperationException(objectPath.toPathString());
        }
        V oldView = viewFactory.apply(oldData);
        V newView = viewFactory.apply(newData);
        if (oldView.equals(newView)) {
            return null;
        } else {
            return eventFactory.apply(clientId, campaignId, adGroupId, oldView, newView);
        }
    }

    private static int parseMultiplier(HierarchicalMultipliersData.Root data) {
        if (StringUtils.isEmpty(data.getMultiplierPct())) {
            return 100;
        } else {
            return Integer.parseInt(data.getMultiplierPct());
        }
    }

    private static MultiplierEventFactory<Integer> singleMultiplierEventFactory(
            OutputCategory campaignCategory,
            OutputCategory adGroupCategory) {
        return (clientId, campaignId, adGroupId, oldView, newView) ->
                new SingleMultiplierEvent(clientId, campaignId, adGroupId,
                        adGroupId == null ? campaignCategory : adGroupCategory,
                        oldView, newView);
    }

    @Override
    protected Map<OutputCategory, String> getTableByCategory() {
        return TABLE_BY_CATEGORY;
    }

    @Override
    protected Map<OutputCategory, Collection<FieldKeyValue>> getFieldsValuesByCategory() {
        return FIELDS_VALUES_BY_CATEGORY;
    }

    @Override
    public List<LogRecord> offer(PeekingIterator<ActionLogRecordWithStats> recordIter) {
        final List<ActionLogRecordWithStats> recordGroup = nextRecordGroup(recordIter);
        if (recordGroup.isEmpty()) {
            return Collections.emptyList();
        }
        final ActionLogRecordWithStats firstRecordWithStats = recordGroup.get(0);
        final ActionLogRecordWithStats lastRecordWithStats = recordGroup.get(recordGroup.size() - 1);
        HierarchicalMultipliersData.Root oldData = deserializeBefore(firstRecordWithStats.getRecord())
                .orElseGet(HierarchicalMultipliersData.Root::new);
        HierarchicalMultipliersData.Root newData = deserializeAfter(lastRecordWithStats.getRecord())
                .orElseGet(HierarchicalMultipliersData.Root::new);
        String type = Optional.ofNullable(newData.getType()).orElse(oldData.getType());
        if (TOGGLEABLE_TYPES.contains(type)) {
            // В случае, если первая запись - INSERT, то getIsEnabled() = null
            if (Objects.equals(oldData.getIsEnabled(), "0")) {
                oldData = new HierarchicalMultipliersData.Root().withType(type);
            }
            // В случае, если последняя запись - DELETE, то getIsEnabled() = null
            if (Objects.equals(newData.getIsEnabled(), "0")) {
                newData = new HierarchicalMultipliersData.Root().withType(type);
            }
        }
        LogEvent result = null;
        if (Objects.equals(type, HierarchicalMultipliersType.demography_multiplier.getLiteral())) {
            result = demographyMultipliersEvent(
                    firstRecordWithStats.getRecord().getPath(), oldData, newData, firstRecordWithStats.getRecord());
        } else if (Objects.equals(type, HierarchicalMultipliersType.mobile_multiplier.getLiteral())) {
            result = mobileMultiplierEvent(firstRecordWithStats.getRecord().getPath(), oldData, newData);
        } else if (Objects.equals(type, HierarchicalMultipliersType.performance_tgo_multiplier.getLiteral())) {
            result = performanceTgoMultiplierEvent(firstRecordWithStats.getRecord().getPath(), oldData, newData);
        } else if (Objects.equals(type, HierarchicalMultipliersType.retargeting_multiplier.getLiteral())) {
            result = retargetingMultipliersEvent(firstRecordWithStats.getRecord().getPath(), oldData, newData);
        } else if (Objects.equals(type, HierarchicalMultipliersType.video_multiplier.getLiteral())) {
            result = videoMultiplierEvent(firstRecordWithStats.getRecord().getPath(), oldData, newData);
        }
        if (result != null) {
            OptionalLong operatorUid = firstRecordWithStats.getRecord().getDirectTraceInfo().getOperatorUid();
            return Collections.singletonList(new LogRecord(
                    firstRecordWithStats.getRecord().getDateTime(),
                    operatorUid.isPresent() ? operatorUid.getAsLong() : null,
                    firstRecordWithStats.getRecord().getGtid(),
                    result,
                    firstRecordWithStats.getStats().getChangeSource(),
                    firstRecordWithStats.getStats().getIsChangedByRecommendation()));
        } else {
            return Collections.emptyList();
        }
    }

    @FunctionalInterface
    private interface MultiplierEventFactory<T> {
        LogEvent apply(ClientId clientId, CampaignId campaignId, @Nullable AdGroupId adGroupId, T oldView, T newView);
    }
}
