package ru.yandex.direct.useractionlog.reader.model;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;

import ru.yandex.direct.useractionlog.AdGroupId;
import ru.yandex.direct.useractionlog.CampaignId;
import ru.yandex.direct.useractionlog.ClientId;

@ParametersAreNonnullByDefault
public class DemographyMultipliersEvent implements LogEvent, AdGroupEvent {
    private final List<DemographyMultiplierView> oldDemographyMultipliers;
    private final List<DemographyMultiplierView> newDemographyMultipliers;
    private final ClientId clientId;
    private final CampaignId campaignId;
    private final AdGroupId adGroupId;

    public <V> DemographyMultipliersEvent(ClientId clientId, CampaignId campaignId, @Nullable AdGroupId adGroupId,
                                          List<DemographyMultiplierView> oldDemographyMultipliers,
                                          List<DemographyMultiplierView> newDemographyMultipliers) {
        this.clientId = clientId;
        this.campaignId = campaignId;
        this.adGroupId = adGroupId;
        this.oldDemographyMultipliers = oldDemographyMultipliers;
        this.newDemographyMultipliers = newDemographyMultipliers;
    }

    @Override
    public @GraphQLNonNull
    long getClientId() {
        return clientId.toLong();
    }

    @Override
    public @GraphQLNonNull
    long getCampaignId() {
        return campaignId.toLong();
    }

    @Nonnull
    public @GraphQLNonNull
    List<DemographyMultiplierView> getOldDemographyMultipliers() {
        return oldDemographyMultipliers;
    }

    @Nonnull
    public @GraphQLNonNull
    List<DemographyMultiplierView> getNewDemographyMultipliers() {
        return newDemographyMultipliers;
    }

    @Nullable
    @Override
    public Long getAdGroupId() {
        return adGroupId == null ? null : adGroupId.toLong();
    }

    @Nonnull
    @Override
    public @GraphQLNonNull
    OutputCategory getCategory() {
        if (adGroupId == null) {
            return OutputCategory.CAMPAIGN_DEMOGRAPHY_MULTIPLIERS;
        } else {
            return OutputCategory.ADGROUP_DEMOGRAPHY_MULTIPLIERS;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof DemographyMultipliersEvent)) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        DemographyMultipliersEvent that = (DemographyMultipliersEvent) o;
        return Objects.equals(oldDemographyMultipliers, that.oldDemographyMultipliers) &&
                Objects.equals(newDemographyMultipliers, that.newDemographyMultipliers) &&
                Objects.equals(adGroupId, that.adGroupId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), oldDemographyMultipliers, newDemographyMultipliers, adGroupId);
    }

    @SuppressWarnings("squid:S00115")
    public enum Gender {
        any,
        female,
        male,
    }

    @SuppressWarnings("squid:S00115")
    public enum Age {
        any,
        age_0_17,
        age_18_24,
        age_25_34,
        age_35_44,
        age_45_inf,  // DIRECT-77321
        age_45_54,
        age_55_inf,
    }

    @ParametersAreNonnullByDefault
    public static class DemographyMultiplierView {
        private final Age age;
        private final Gender gender;
        private final int multiplier;

        public DemographyMultiplierView(Age age, Gender gender, int multiplier) {
            this.gender = gender;
            this.age = age;
            this.multiplier = multiplier;
        }

        public @GraphQLNonNull
        Gender getGender() {
            return gender;
        }

        public @GraphQLNonNull
        int getMultiplier() {
            return multiplier;
        }

        public @GraphQLNonNull
        Age getAge() {
            return age;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            DemographyMultiplierView that = (DemographyMultiplierView) o;
            return multiplier == that.multiplier &&
                    age == that.age &&
                    gender == that.gender;
        }

        @Override
        public int hashCode() {
            return Objects.hash(age, gender, multiplier);
        }
    }

}

