package ru.yandex.direct.useractionlog.writer;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.github.shyiko.mysql.binlog.GtidSet;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.reader.BinlogReader;
import ru.yandex.direct.binlog.reader.BinlogSource;
import ru.yandex.direct.binlog.reader.BinlogStateOptimisticSnapshotter;
import ru.yandex.direct.binlog.reader.EnrichedEvent;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.StateBound;
import ru.yandex.direct.binlog.reader.StatefulBinlogSource;
import ru.yandex.direct.binlog.reader.Transaction;
import ru.yandex.direct.binlog.reader.TransactionReader;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.mysql.BinlogEventData;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.mysql.MySQLServerBuilder;
import ru.yandex.direct.mysql.MySQLSimpleConnector;
import ru.yandex.direct.useractionlog.db.ActionLogWriteRepository;
import ru.yandex.direct.useractionlog.db.StateReaderWriter;
import ru.yandex.direct.useractionlog.db.UserActionLogStates;
import ru.yandex.direct.useractionlog.dict.DictRepository;
import ru.yandex.direct.useractionlog.dict.DictRequest;
import ru.yandex.direct.useractionlog.dict.DictResponsesAccessor;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.writer.generator.BatchRowDictProcessing;
import ru.yandex.direct.useractionlog.writer.generator.RowProcessingStrategy;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.db.MySQLConnector;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;

@ParametersAreNonnullByDefault
public class ActionProcessor implements Interrupts.InterruptibleCheckedRunnable {
    /**
     * См. {@link #defaultEventBatchSize }
     */
    static final int DEFAULT_EVENT_BATCH_SIZE = 5_000;

    /**
     * См. {@link #recordBatchSize}
     */
    static final int DEFAULT_RECORD_BATCH_SIZE = 50_000;

    /**
     * См. {@link #maxBufferedEvents}
     **/
    static final int DEFAULT_MAX_BUFFERED_EVENTS = 100;

    private static final Logger logger = LoggerFactory.getLogger(ActionProcessor.class);
    /**
     * Сколько ждать перед перезапуском, если соединение с mysql было разорвано в связи с тайм-аутом или окончанием
     * бинлога.
     */
    private static final Duration RECONNECT_PAUSE = Duration.ofSeconds(30);
    /**
     * Название источника. Например, "ppc:1".
     */
    private final String source;

    /**
     * Если в бинлоге не было новых событий в течение этого периода, следует переустановить соединение с бинлогом.
     */
    private final Duration binlogKeepAliveTimeout;

    /**
     * Максимальное время, на которое можно откладывать запись новых событий и словарных данных.
     * По истечении этого времени в базу записывается всё, что успело накопиться.
     */
    private final Duration batchDuration;

    /**
     * Максимальное количество binlog-событий, которое можно накопить перед началом пакетной обработки.
     */
    private final int defaultEventBatchSize;

    /**
     * Максимальное количество записей таблицы пользовательских логов, которое можно накопить перед записью в БД.
     */
    private final int recordBatchSize;

    /**
     * Размер очереди событий из бинлога. Эта очередь пополняется из параллельного потока, чтобы соединение
     * с mysql реже рвалось из-за простоя.
     */
    private final int maxBufferedEvents;

    /**
     * Если rowProcessingStrategy бросает ошибку при обработке определённого события, то можно упасть,
     * а можно записать ошибку в лог и продолжить работать дальше.
     */
    private final ErrorWrapper errorWrapper;

    @Nullable
    private final Integer initialServerId;
    private final List<EnrichedRow> enrichedRowsBuffer;
    private final List<ActionLogRecord> recordsBuffer;
    private final BufferedDictRepository bufferedDictRepository;
    @Nullable
    private final String untilGtidSet;
    private final MySQLConnector mysqlConnector;
    private final MySQLServerBuilder schemaReplicaMysqlBuilder;
    private final StateReaderWriter readWriteStateTable; // TODO(lagunov) rename
    private final RowProcessingStrategy rowProcessingStrategy;
    @Nullable
    private final ActionLogWriteRepository writeActionLogTable;  // TODO(lagunov) rename
    @Nullable
    private final Semaphore binlogStateFetchingSemaphore;
    private final Set<String> gtidIgnoreSet;

    @Nullable
    private final PpcProperty<Integer> eventBatchSizeProperty;

    @Nullable
    private final Semaphore schemaReplicaMysqlSemaphore;
    private MySQLBinlogState lastWrittenState = null;

    // Конструктор с большим количеством аргументом. Прикрывается билдером.
    @SuppressWarnings({"squid:S00107", "checkstyle:parameternumber"})
    private ActionProcessor(
            boolean skipErroneousEvents,
            DirectConfig directConfig,
            DbConfig dbConfig,
            DictRepository dictRepository,
            Duration batchDuration,
            Duration binlogKeepAliveTimeout,
            int defaultEventBatchSize,
            int maxBufferedEvents,
            int recordBatchSize,
            MySQLServerBuilder schemaReplicaMysqlBuilder,
            StateReaderWriter readWriteStateTable,
            RowProcessingStrategy rowProcessingStrategy,
            @Nullable Integer initialServerId,
            @Nullable Semaphore binlogStateFetchingSemaphore,
            @Nullable Semaphore schemaReplicaMysqlSemaphore,
            @Nullable String untilGtidSet,
            @Nullable ActionLogWriteRepository writeActionLogTable,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.binlogKeepAliveTimeout = binlogKeepAliveTimeout;
        this.batchDuration = batchDuration;
        this.defaultEventBatchSize = defaultEventBatchSize;
        this.recordBatchSize = recordBatchSize;
        this.maxBufferedEvents = maxBufferedEvents;
        this.initialServerId = initialServerId;
        this.untilGtidSet = untilGtidSet;
        this.schemaReplicaMysqlBuilder = schemaReplicaMysqlBuilder;
        this.readWriteStateTable = readWriteStateTable;
        this.rowProcessingStrategy = rowProcessingStrategy;
        this.writeActionLogTable = writeActionLogTable;
        this.binlogStateFetchingSemaphore = binlogStateFetchingSemaphore;
        this.schemaReplicaMysqlSemaphore = schemaReplicaMysqlSemaphore;

        errorWrapper = new ErrorWrapper(skipErroneousEvents);
        bufferedDictRepository = new BufferedDictRepository(dictRepository);
        mysqlConnector = new MySQLSimpleConnector(
                dbConfig.getHosts().get(0),
                dbConfig.getPort(),
                dbConfig.getUser(),
                dbConfig.getPass());
        recordsBuffer = new ArrayList<>();
        enrichedRowsBuffer = new ArrayList<>();
        source = dbConfig.getDbName();
        gtidIgnoreSet = new HashSet<>(directConfig.getStringList("alw.gtid_ignore_list"));
        eventBatchSizeProperty = ppcPropertiesSupport != null
                ? ppcPropertiesSupport.get(PpcPropertyNames.ALW_EVENT_BATCH_SIZE, Duration.ofSeconds(10))
                : null;
    }

    /**
     * @return true - fromGtidSet не достиг untilGtidSet, т.е. {@literal fromGtidSet < untilGtidSet}
     */
    static boolean finalStateNotReached(String fromGtidSet, @Nullable String untilGtidSet) {
        return untilGtidSet == null || !new GtidSet(untilGtidSet).isContainedWithin(new GtidSet(fromGtidSet));
    }

    /**
     * @return true - fromGtidSet обогнал untilGtidSet, т.е. {@literal fromGtidSet > untilGtidSet}
     */
    static boolean finalStateOvertaken(String fromGtidSet, @Nullable String untilGtidSet) {
        return !(untilGtidSet == null || new GtidSet(fromGtidSet).isContainedWithin(new GtidSet(untilGtidSet)));
    }

    private static <T> T runWithSemaphore(@Nullable Semaphore semaphore, Interrupts.InterruptibleSupplier<T> supplier)
            throws InterruptedException {
        if (semaphore != null) {
            semaphore.acquire();
        }
        try {
            return supplier.get();
        } finally {
            if (semaphore != null) {
                semaphore.release();
            }
        }
    }

    /**
     * @param unprocessed Список пар. Первый элемент пары - бинлог-кортеж, для которого не были найдены запрошенные
     *                    словарные значения. Второй элемент пары - сами необработанные словарные запросы.
     */
    public static void logUnprocessedRequests(List<Pair<EnrichedRow, Collection<DictRequest>>> unprocessed) {
        List<DictRequest> unprocessedRequests = unprocessed.stream()
                .map(Pair::getRight)
                .flatMap(Collection::stream)
                .sorted(Comparator
                        .comparing(DictRequest::getCategory)
                        .thenComparingLong(DictRequest::getId))
                .collect(Collectors.toList());

        // В DIRECT-67483 была идея записывать эти события в отдельную таблицу, но, кажется,
        // никакой практической ценности это нововведение не даст.
        if (logger.isErrorEnabled() && !unprocessedRequests.isEmpty()) {
            logger.error("Can't handle {} rows due to absent dictionary data for {}",
                    unprocessed.size(),
                    unprocessedRequests.stream()
                            .collect(Collectors.groupingBy(DictRequest::getCategory))
                            .entrySet()
                            .stream()
                            .map(e -> String.format("%s{%s}", e.getKey(), e.getValue().stream()
                                    .map(r -> Long.toString(r.getId()))
                                    .collect(Collectors.toSet())
                                    .stream()
                                    .sorted()
                                    .collect(Collectors.joining(", "))))
                            .sorted()
                            .collect(Collectors.joining(" ")));
        }
    }

    @Override
    public void run() throws InterruptedException {
        try {
            UserActionLogStates states;
            boolean continueWorking;
            do {
                states = runWithSemaphore(binlogStateFetchingSemaphore,
                        () -> readWriteStateTable.read.getActualStates(source));
                if ((states.getDict() == null) != (states.getLog() == null)) {
                    // Если не записан никакой стейт, скорее всего приложение запускается на dev-среде с пустой базой.
                    // Синхронизация будет корректно работать только для кампаний, созданных после такого запуска.
                    // Если есть стейт логов и словаря, значит база прошла процедуру InitDictionary
                    // Если записан только один стейт из двух, то это какая-то некорректная база, непонятно что с ней
                    // делать.
                    throw new IllegalStateException(String.format("Log state is%s empty, but dict state is%s empty.",
                            states.getLog() == null ? "" : " NOT",
                            states.getDict() == null ? "" : " NOT"));
                }
                if (states.getLog() == null) {
                    MySQLBinlogState state = new BinlogStateOptimisticSnapshotter(makeServerId())
                            .snapshot(mysqlConnector);
                    logger.warn("Log state is empty. Start writing logs from @@global.gtid_executed = {}",
                            state.getGtidSet());
                    states = UserActionLogStates.builder()
                            .withDict(state)
                            .withLog(state)
                            .build();
                    // Если запустили на dev-среде, то стейт может быть пустым. Далее приложение может упасть
                    // после записи первого стейта логов, но до записи первого стейта словаря. Такую базу невозможно
                    // будет восстановить. Поэтому сразу в базу записывается полный корректный стейт.
                    readWriteStateTable.write.saveBothStates(source, state);
                }
                Objects.requireNonNull(states.getLog());
                Objects.requireNonNull(states.getDict());
                GtidSet logGtidSet = new GtidSet(states.getLog().getGtidSet());
                GtidSet dictGtidSet = new GtidSet(states.getDict().getGtidSet());
                Preconditions.checkState(dictGtidSet.isContainedWithin(logGtidSet),
                        "Fatal error. gtid_set of dict is greater than gtid_set of log.");

                continueWorking = !Thread.currentThread().isInterrupted()
                        && finalStateNotReached(states.getDict().getGtidSet(), untilGtidSet)
                        && !handleBinlog(states, makeServerId());
            } while (continueWorking);
        } catch (InterruptedException err) {
            Thread.currentThread().interrupt();
            throw err;
        }
    }

    private boolean handleBinlog(UserActionLogStates states, int currentServerId) throws InterruptedException {
        Objects.requireNonNull(states.getLog());
        Objects.requireNonNull(states.getDict());
        try (BinlogReader binlogReader = new BinlogReader(
                new StatefulBinlogSource(
                        new BinlogSource(source, mysqlConnector),
                        states.getDict()),
                schemaReplicaMysqlBuilder.copy(),
                schemaReplicaMysqlSemaphore,
                binlogKeepAliveTimeout,
                currentServerId,
                maxBufferedEvents)) {
            // DIRECT-75522
            //
            // У логов и у словаря есть своё состояние.
            // Так как в кликхаусе невозможно атомарно записать сразу несколько таблиц,
            // то используется такой хитрый алгоритм:
            //
            // 1. Записывается новые записи логов
            // 2. Записывается состояние логов
            // 3. Записываются новые словарные данные
            // 4. Записывается состояние словаря
            //
            // Предположим, приложение упало в процессе шага 1 или между шагом 1 и 2. В таблице логов новые данные, а
            // состояние логов осталось старым. Тогда при перезапуске приложение возьмёт старое состояние и повторно
            // сгенерирует точно такие же логи, как и уже записанные. Дублирование записей в таблице логов ничего не
            // ломает.
            //
            // Предположим, приложение упало между шагом 2 и 3, в процессе шага 3 или между шагом 3 и 4. Логи свежие, а
            // словарь целиком или частично старый. Тогда при перезапуске приложение обработает бинлог между dictGtidSet
            // и logGtidSet. Благодаря тому, что генерация словарных данных - идемпотентная операция, при достижении
            // logGtidSet словарь придёт в консистентное состояние. При этом в процессе прогона между dictGtidSet и
            // logGtidSet ничего не пишется в таблицу логов, иначе получаемые логи могут содержать словарные данные из
            // будущего.
            //
            // Поменять местами запись логов и словаря нельзя. Если бы сначала записывался словарь, а потом логи,
            // то при падении между двумя записями и последующем рестарте логи брали бы из словаря данные из будущего.
            handleOneBinlogConnection(states.getDict(), states.getLog().getGtidSet(), false, binlogReader);
            handleOneBinlogConnection(states.getLog(), untilGtidSet, true, binlogReader);
            return true;
        } catch (RuntimeTimeoutException ignored) {
            logger.info("Read from binlog timed out. Sleep {} before reconnect.", RECONNECT_PAUSE);
            Thread.sleep(RECONNECT_PAUSE.toMillis());
        } catch (RuntimeException e1) {
            throw new Checked.CheckedException("Binlog processing failed for source: " + source, e1);
        }
        return false;
    }

    /**
     * Обрабатывает на одном binlog-соединении полуинтервал (currentState, untilGtidSet]
     *
     * @param currentState Последний обработанный gtid_set + схема, на котором остановился синхронизатор.
     * @param untilGtidSet После обработки этого gtid_set следует остановиться. Если null, то остановится лишь при
     *                     прерывании потока.
     * @param writeLogs    true - в базу будут записываться и логи, и словарные данные. false - только словарь.
     *                     Если нужен только словарь, то могут быть применены оптимизации,
     *                     см. {@link ru.yandex.direct.useractionlog.writer.generator.PureDictFillerFactory}.
     * @param binlogReader Источник binlog-событий.
     */
    private void handleOneBinlogConnection(MySQLBinlogState currentState,
                                           @Nullable String untilGtidSet,
                                           boolean writeLogs,
                                           BinlogReader binlogReader)
            throws InterruptedException {
        try {
            logger.debug("Start handling one binlog connection from state {} until state {}, with{} writing logs",
                    currentState.getGtidSet(),
                    untilGtidSet,
                    writeLogs ? "" : "OUT");
            MonotonicTime nextFlushDataTime = NanoTimeClock.now().plus(batchDuration);
            TransactionReader transactionReader = new TransactionReader(binlogReader);
            while (!Thread.currentThread().isInterrupted() && finalStateNotReached(
                    transactionReader.getState().getState().getGtidSet(), untilGtidSet)) {
                Pair<MySQLBinlogState, List<EnrichedEvent>> stateAndEvents =
                        readTransactionsBatch(transactionReader, untilGtidSet, nextFlushDataTime,
                                recordBatchSize - enrichedRowsBuffer.size());
                logger.debug("Fetched {} enriched events, got new state {}",
                        stateAndEvents.getRight().size(),
                        stateAndEvents.getLeft().getGtidSet());
                currentState = stateAndEvents.getLeft();
                var rows = makeRowsFromEvents(stateAndEvents.getRight());
                enrichedRowsBuffer.addAll(rows);
                if (nextFlushDataTime.isAtOrBefore(NanoTimeClock.now()) ||
                        enrichedRowsBuffer.size() >= recordBatchSize) {
                    boolean updated = flushDataIfHasSome(currentState, writeLogs);
                    if (updated) {
                        nextFlushDataTime = NanoTimeClock.now().plus(batchDuration);
                    } else {
                        // Если попали сюда, значит от источника приходит мало событий либо они все нерелевантные.
                        // Если не поставить никакой тайм-аут, то приложение войдёт в холостой цикл, в котором будет
                        // делать кучу пустых вычислений и писать кучу записей в лог.
                        nextFlushDataTime = NanoTimeClock.now().plus(Duration.ofSeconds(3));
                    }
                } else {
                    logger.debug("Remained either {} (of {} max) log records"
                                    + " or {} (of {} max) seconds until buffer flush",
                            recordBatchSize - enrichedRowsBuffer.size(), recordBatchSize,
                            nextFlushDataTime.minus(NanoTimeClock.now()).toMillis() / 1e3,
                            batchDuration.toMillis() / 1e3);
                }
            }
        } finally {
            if (!Thread.currentThread().isInterrupted()) {
                flushDataIfHasSome(currentState, writeLogs);
            }
        }
    }

    /**
     * DIRECT-78909
     * <p>
     * У каждого соединения с mysql должен быть свой уникальный идентификатор. Если к одному mysql подключится
     * два binlog-клиента с одинаковым server id, то один из них в итоге будет убит, причём может быть убит не сразу.
     * <p>
     * См. https://github.com/percona/percona-server/blob/5.7/sql/rpl_master.cc#L495-L505
     */
    private int makeServerId() {
        if (initialServerId == null) {
            return ThreadLocalRandom.current().nextInt(1 << 15) << 16;
        } else {
            return initialServerId;
        }
    }

    /**
     * Превращает поток бинлог-событий в пользовательские логи и новые словарные данные, и сохраняет в буферы
     * {@link #recordsBuffer} и {@link #bufferedDictRepository}.
     *
     * @param rows     Строки бинлог-событий
     * @param needLogs true - будут генерироваться и логи, и словарные данные. false - только словарные данные.
     */
    void handleRows(List<EnrichedRow> rows, boolean needLogs) {
        BatchRowDictProcessing.Result result;
        MonotonicTime startTime = NanoTimeClock.now();
        if (needLogs) {
            result = BatchRowDictProcessing.handleEvents(
                    bufferedDictRepository, rowProcessingStrategy,
                    rows,
                    errorWrapper);
            for (Pair<EnrichedRow, DictResponsesAccessor> pair : result.processed) {
                EnrichedRow row = pair.getLeft();
                DictResponsesAccessor dictResponseAccessor = pair.getRight();
                errorWrapper.accept(row, () ->
                        recordsBuffer.addAll(rowProcessingStrategy.processEvent(row, dictResponseAccessor)));
            }
        } else {
            result = BatchRowDictProcessing.handleEvents(
                    bufferedDictRepository, rowProcessingStrategy.makePureDictFiller(),
                    rows,
                    errorWrapper);
        }
        logger.info("Handled {} enriched rows in {} seconds",
                rows.size(),
                NanoTimeClock.now().minus(startTime).toMillis() / 1e3);
        if (!result.unprocessed.isEmpty()) {
            logUnprocessedRequests(result.unprocessed);
        }
    }

    List<EnrichedRow> makeRowsFromEvents(List<EnrichedEvent> events) {
        List<EnrichedRow> rows = new ArrayList<>();
        for (EnrichedEvent event : events) {
            if (!gtidIgnoreSet.contains(event.getGtid())) {
                event.rowsStream().forEachOrdered(rows::add);
            } else {
                logger.warn("Skipped event {} due to gtidIgnoreSet", event.getGtid());
            }
        }
        return rows;
    }

    /**
     * Предназначено для юнит-тестов, чтобы не разворачивать кликхаус
     */
    ImmutableList<ActionLogRecord> makeBufferCopy() {
        return ImmutableList.copyOf(recordsBuffer);
    }

    /**
     * Записать в БД содержимое буферов логов и новых словарных данных, затем очистить буферы.
     *
     * @param newState  Какому состоянию соответствуют данные.
     * @param writeLogs true - сбросить буфер логов, затем записать состояние логов, затем буфер словаря, затем
     *                  состояние словаря. false - только буфер словаря и состояние словаря.
     * @return true - если хоть что-нибудь было записано. false - если нужные буферы были пусты и ничего не было
     * записано.
     */
    private boolean flushDataIfHasSome(MySQLBinlogState newState, boolean writeLogs) {
        boolean updatedLogs = false;
        boolean updatedDict = false;
        if (!Objects.equals(lastWrittenState, newState)) {
            // Инвариант: Lgtid >= Dgtid
            // Возможны состояния:
            // * Есть новые логи и новый словарь - сначала запись логов, потом запись словаря
            // * Есть новые логи и нет нового словаря - аналогично, пусть пишет пустой словарь
            // * Нет логов, есть словарь - аналогично, пусть пишет пустые логи
            // * Нет логов и нет словаря - пусть пишет пустой стейт.
            for (var chunk : Iterables.partition(enrichedRowsBuffer, recordBatchSize)) {
                handleRows(chunk, writeLogs);
                if (writeLogs) {
                    updatedLogs = writeLogs() || updatedLogs;
                }
            }
            readWriteStateTable.write.saveLogState(source, newState);
            updatedDict = writeDict();
            readWriteStateTable.write.saveDictState(source, newState);

            lastWrittenState = newState;
        }

        enrichedRowsBuffer.clear();

        return updatedLogs || updatedDict;
    }

    private boolean writeDict() {
        boolean updatedDict;
        int bufferSize = bufferedDictRepository.bufferSize();
        updatedDict = bufferSize > 0;
        if (updatedDict) {
            MonotonicTime startTime = NanoTimeClock.now();
            bufferedDictRepository.flush();
            logger.info("Written {} dict records in {} seconds",
                    bufferSize,
                    NanoTimeClock.now().minus(startTime).toMillis() / 1e3);
        } else {
            logger.info("No new dict records, writing only state");
        }
        return updatedDict;
    }

    private boolean writeLogs() {
        boolean updatedLogs;
        updatedLogs = !recordsBuffer.isEmpty();
        if (updatedLogs) {
            Objects.requireNonNull(writeActionLogTable);
            MonotonicTime startTime = NanoTimeClock.now();
            writeActionLogTable.insert(recordsBuffer);
            recordsBuffer.clear();
            logger.info("Written {} action log records in {} seconds",
                    recordsBuffer.size(),
                    NanoTimeClock.now().minus(startTime).toMillis() / 1e3);
        } else {
            logger.info("No new log records, writing only state");
        }
        return updatedLogs;
    }

    /**
     * Получить пачку транзакций.
     *
     * @param transactionReader Источник бинлог-событий.
     * @param untilGtidSet      До какого gtid_set включительно следует читать.
     * @param readUntilTime     До какого времени следует получать события.
     * @return Пара. Первый элемент - стейт, который соответствует последнему событию в возвращаемом списке, или текущий
     * стейт, если список пуст. Второй элемент - список бинлог-событий.
     */
    @Nonnull
    private Pair<MySQLBinlogState, List<EnrichedEvent>> readTransactionsBatch(
            TransactionReader transactionReader,
            @Nullable String untilGtidSet,
            MonotonicTime readUntilTime,
            long rowsSoftLimit
    ) throws InterruptedException {
        var batchSize = Optional.ofNullable(eventBatchSizeProperty)
                .map(PpcProperty::get)
                .orElse(defaultEventBatchSize);

        List<EnrichedEvent> events = new ArrayList<>(batchSize);
        MySQLBinlogState currentState = transactionReader.getState().getState();
        AtomicLong rowsCount = new AtomicLong(0);

        Duration currentReadTimeout;
        try {
            while (!Thread.currentThread().isInterrupted()
                    && events.size() < batchSize
                    && (currentReadTimeout = readUntilTime.minus(NanoTimeClock.now())).compareTo(Duration.ZERO) > 0
                    && finalStateNotReached(currentState.getGtidSet(), untilGtidSet)
                    && rowsCount.get() < rowsSoftLimit
            ) {
                // Все кортежи в transaction относятся к одному и тому же gtid_set
                StateBound<Transaction> transaction = transactionReader.readTransaction(currentReadTimeout);
                currentState = transaction.getStateSet().iterator().next().getState();
                transaction.getData().getEnrichedEvents().forEachOrdered(e -> {
                            events.add(e);
                            BinlogEventData data = e.getEvent().getData();
                            if (data instanceof BinlogEventData.Insert) {
                                rowsCount.addAndGet(((BinlogEventData.Insert) data).getRows().size());
                            } else if (data instanceof BinlogEventData.Update) {
                                rowsCount.addAndGet(((BinlogEventData.Update) data).getRows().size());
                            } else if (data instanceof BinlogEventData.Delete) {
                                rowsCount.addAndGet(((BinlogEventData.Delete) data).getRows().size());
                            }
                        }
                );
            }
        } catch (RuntimeTimeoutException ex) {
            if (events.isEmpty()) {
                throw ex;
            }
        }
        return Pair.of(currentState, events);
    }

    public static class Builder {
        private boolean binlogStateFetchingSemaphoreSet;
        private boolean schemaReplicaMysqlSemaphoreSet;
        private boolean writeActionLogTableSet;
        private DirectConfig directConfig;
        private DbConfig dbConfig;
        private DictRepository dictRepository;
        private Duration binlogKeepAliveTimeout;
        private Duration batchDuration;
        private int eventBatchSize = DEFAULT_EVENT_BATCH_SIZE;
        private int recordBatchSize = DEFAULT_RECORD_BATCH_SIZE;
        private Integer initialServerId;
        private int maxBufferedEvents = DEFAULT_MAX_BUFFERED_EVENTS;
        private MySQLServerBuilder schemaReplicaMysqlBuilder;
        private StateReaderWriter readWriteStateTable;
        private RowProcessingStrategy rowProcessingStrategy;
        private ActionLogWriteRepository writeActionLogTable;
        private String untilGtidSet;
        private Semaphore binlogStateFetchingSemaphore;
        private Semaphore schemaReplicaMysqlSemaphore;
        private boolean skipErroneousEvents = false;
        private PpcPropertiesSupport ppcPropertiesSupport = null;

        public Builder withDirectConfig(DirectConfig directConfig) {
            this.directConfig = directConfig;
            return this;
        }

        /**
         * Информация об источнике бинлога
         */
        public Builder withDbConfig(DbConfig dbConfig) {
            this.dbConfig = dbConfig;
            return this;
        }

        /**
         * Настройки для временного процесса mysqld, на котором будут обкатываться DDL-запросы.
         */
        public Builder withSchemaReplicaMysqlBuilder(MySQLServerBuilder schemaReplicaMysqlBuilder) {
            this.schemaReplicaMysqlBuilder = schemaReplicaMysqlBuilder;
            return this;
        }

        /**
         * У каждого соединения с mysql должен быть свой уникальный идентификатор. Если к одному mysql подключится
         * два binlog-клиента с одинаковым server id, то один из них в итоге будет убит, причём
         * может быть убит не сразу.
         * <p>
         * См. https://github.com/percona/percona-server/blob/5.7/sql/rpl_master.cc#L495-L505
         * <p>
         * По умолчанию выбирается случайный server id при каждом установлении нового соединения с источником бинлога.
         */
        public Builder withInitialServerId(@Nullable Integer initialServerId) {
            this.initialServerId = initialServerId;
            return this;
        }

        /**
         * Если в бинлоге не было новых событий в течение этого периода, следует переустановить соединение с бинлогом.
         */
        public Builder withBinlogKeepAliveTimeout(Duration binlogKeepAliveTimeout) {
            this.binlogKeepAliveTimeout = binlogKeepAliveTimeout;
            return this;
        }

        /**
         * Размер очереди на чтение бинлог-событий
         */
        public Builder withMaxBufferedEvents(int maxBufferedEvents) {
            this.maxBufferedEvents = maxBufferedEvents;
            return this;
        }

        /**
         * Максимальное количество транзакций, которые следует пакетно обрабатывать.
         */
        public Builder withEventBatchSize(int eventBatchSize) {
            Preconditions.checkState(eventBatchSize > 0, "Wrong eventBatchSize");
            this.eventBatchSize = eventBatchSize;
            return this;
        }

        /**
         * Размер буфера на запись строк в таблицу логов.
         */
        public Builder withRecordBatchSize(int recordBatchSize) {
            Preconditions.checkState(recordBatchSize > 0, "Wrong recordBatchSize");
            this.recordBatchSize = recordBatchSize;
            return this;
        }

        /**
         * Максимальное время, которое можно потратить на сбор транзакций для их пакетной обработки.
         */
        public Builder withBatchDuration(Duration batchDuration) {
            this.batchDuration = batchDuration;
            return this;
        }

        /**
         * Объект для чтения и записи словарных данных.
         */
        public Builder withDictRepository(DictRepository dictRepository) {
            this.dictRepository = dictRepository;
            return this;
        }

        /**
         * Объект для чтения и записи стейтов.
         */
        // TODO rename
        public Builder withReadWriteStateTable(StateReaderWriter readWriteStateTable) {
            this.readWriteStateTable = readWriteStateTable;
            return this;
        }

        /**
         * Объект для трансформации бинлог-событий в пользовательские логи и словарные данные.
         */
        public Builder withRowProcessingStrategy(RowProcessingStrategy rowProcessingStrategy) {
            this.rowProcessingStrategy = rowProcessingStrategy;
            return this;
        }

        /**
         * Объект для записи пользовательских логов.
         */
        public Builder withWriteActionLogTable(@Nullable ActionLogWriteRepository writeActionLogTable) {
            writeActionLogTableSet = true;
            this.writeActionLogTable = writeActionLogTable;
            return this;
        }

        /**
         * До какого gtidSet НЕ включительно следует обрабатывать бинлог-события. Если не указан, то обрабатывать
         * до бесконечности.
         */
        public Builder withUntilGtidSet(@Nullable String untilGtidSet) {
            this.untilGtidSet = untilGtidSet;
            return this;
        }

        /**
         * Ограничитель на одновременное получение стейтов. Например, на MDB (который DBaaS, который
         * кликхаус-как-сервис) по умолчанию стоит жёсткое ограничение - не более 512 Мб оперативной памяти на кликхаус.
         * Если приложение пытается одновременно получить состояние для 15 шардов ppc, то часть запросов может упасть.
         */
        public Builder withBinlogStateFetchingSemaphore(@Nullable Semaphore binlogStateFetchingSemaphore) {
            binlogStateFetchingSemaphoreSet = true;
            this.binlogStateFetchingSemaphore = binlogStateFetchingSemaphore;
            return this;
        }

        /**
         * Ограничитель на одновременную синхронизацию схемы базы с временным mysqld.
         */
        public Builder withSchemaReplicaMysqlSemaphore(@Nullable Semaphore schemaReplicaMysqlSemaphore) {
            schemaReplicaMysqlSemaphoreSet = true;
            this.schemaReplicaMysqlSemaphore = schemaReplicaMysqlSemaphore;
            return this;
        }

        public Builder withPpcPropertiesSupport(PpcPropertiesSupport ppcPropertiesSupport) {
            this.ppcPropertiesSupport = ppcPropertiesSupport;
            return this;
        }

        public ActionProcessor build() {
            Objects.requireNonNull(batchDuration, "Forgotten batchDuration");
            Objects.requireNonNull(directConfig, "Forgotten directConfig");
            Objects.requireNonNull(dbConfig, "Forgotten dbConfig");
            Objects.requireNonNull(dictRepository, "Forgotten dictRepository");
            Objects.requireNonNull(readWriteStateTable, "Forgotten readWriteStateTable");
            Objects.requireNonNull(rowProcessingStrategy, "Forgotten rowProcessingStrategy");
            Objects.requireNonNull(schemaReplicaMysqlBuilder, "Forgotten schemaReplicaMysqlBuilder");
            Preconditions.checkState(binlogStateFetchingSemaphoreSet, "Forgotten binlogStateFetchingSemaphore");
            Preconditions.checkState(schemaReplicaMysqlSemaphoreSet, "Forgotten schemaReplicaMysqlSemaphore");
            Preconditions.checkState(writeActionLogTableSet, "Forgotten writeActionLogTable");

            Preconditions.checkState(batchDuration.compareTo(Duration.ZERO) > 0, "Expected positive batchDuration");
            Preconditions.checkState(eventBatchSize > 0, "Expected positive eventBatchSize");
            Preconditions.checkState(maxBufferedEvents > 0, "Expected positive maxBufferedEvents");

            return new ActionProcessor(
                    skipErroneousEvents,
                    directConfig,
                    dbConfig,
                    dictRepository,
                    batchDuration,
                    binlogKeepAliveTimeout,
                    eventBatchSize,
                    maxBufferedEvents,
                    recordBatchSize,
                    schemaReplicaMysqlBuilder,
                    readWriteStateTable,
                    rowProcessingStrategy,
                    initialServerId,
                    binlogStateFetchingSemaphore,
                    schemaReplicaMysqlSemaphore,
                    untilGtidSet,
                    writeActionLogTable,
                    ppcPropertiesSupport
                    );
        }

        public Builder withSkipErroneousEvents(boolean skipErroneousEvents) {
            this.skipErroneousEvents = skipErroneousEvents;
            return this;
        }
    }
}
