package ru.yandex.direct.useractionlog.writer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;

import ru.yandex.direct.useractionlog.dict.DictRepository;
import ru.yandex.direct.useractionlog.dict.DictRequest;

/**
 * Отложенная запись всех новых словарных данных.
 */
@ParametersAreNonnullByDefault
public class BufferedDictRepository implements DictRepository {
    private static final Object EMPTY = new Object();
    private final DictRepository forward;
    private Map<DictRequest, Object> newData = new HashMap<>();

    public BufferedDictRepository(DictRepository forward) {
        Preconditions.checkState(!(forward instanceof BufferedDictRepository),
                "Chaining of BufferedDictRepository is not supported");
        this.forward = forward;
    }

    /**
     * {@inheritDoc}
     * <p>
     * Если в буфере есть ответы на запрошенные данные, то такие запросы не пойдут дальше
     * по цепочке {@link DictRepository}.
     */
    @Nonnull
    @Override
    public Map<DictRequest, Object> getData(Collection<DictRequest> dictRequests) {
        Collection<DictRequest> forwardRequests = new ArrayList<>(dictRequests.size());
        Map<DictRequest, Object> result = Maps.newHashMapWithExpectedSize(dictRequests.size());
        for (DictRequest dictRequest : dictRequests) {
            Object changedResult = newData.getOrDefault(dictRequest, EMPTY);
            if (changedResult == EMPTY) {
                forwardRequests.add(dictRequest);
            } else {
                result.put(dictRequest, changedResult);
            }
        }
        if (!forwardRequests.isEmpty()) {
            result.putAll(forward.getData(forwardRequests));
        }
        return result;
    }

    @Override
    public void addData(Map<DictRequest, Object> data) {
        newData.putAll(data);
    }

    public int bufferSize() {
        return newData.size();
    }

    /**
     * Записать всё содержимое буфера и очистить буфер.
     */
    public void flush() {
        forward.addData(newData);
        newData.clear();
    }
}
