package ru.yandex.direct.useractionlog.writer;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import ru.yandex.direct.binlog.reader.BinlogStateSet;
import ru.yandex.direct.binlog.reader.EnrichedEvent;
import ru.yandex.direct.binlog.reader.StateBound;
import ru.yandex.direct.binlog.reader.Transaction;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;

public class EnrichedEventReader {

    private Interrupts.InterruptibleFunction<Duration, StateBound<Transaction>> transactionSupplier;
    private int minBatchSize;
    private final Duration maxBatchDuration;
    private final Duration readTimeout;

    public EnrichedEventReader(
            Interrupts.InterruptibleFunction<Duration, StateBound<Transaction>> transactionSupplier,
            int minBatchSize,
            Duration maxBatchDuration,
            Duration readTimeout) {
        this.transactionSupplier = transactionSupplier;
        this.minBatchSize = minBatchSize;
        this.maxBatchDuration = maxBatchDuration;
        this.readTimeout = readTimeout;
    }

    public StateBound<List<EnrichedEvent>> readEnrichedEventBatch() throws InterruptedException {
        TimeoutCalc timeoutCalc = new TimeoutCalc();

        BinlogStateSet stateSet = new BinlogStateSet();
        List<EnrichedEvent> result = new ArrayList<>(minBatchSize);
        while (result.size() < minBatchSize) {
            try {
                StateBound<Transaction> transaction = transactionSupplier.apply(timeoutCalc.timeout());
                transaction.getData()
                        .getEnrichedEvents()
                        .forEachOrdered(result::add);
                stateSet.addAll(transaction.getStateSet());
                timeoutCalc.processedResult();
            } catch (RuntimeTimeoutException ex) {
                if (result.isEmpty()) {
                    throw ex;
                } else {
                    break;
                }
            }
        }
        return new StateBound<>(stateSet, result);
    }

    private class TimeoutCalc {
        private final long startTime = System.nanoTime();
        long firstResultTime = 0;

        Duration timeout() {
            long timeout = firstResultTime != 0
                    ? firstResultTime + maxBatchDuration.toNanos() - System.nanoTime()
                    : startTime + readTimeout.toNanos() - System.nanoTime();
            return Duration.ofNanos(Long.max(0, timeout));
        }

        void processedResult() {
            firstResultTime = firstResultTime != 0 ? firstResultTime : System.nanoTime();
        }
    }
}
