package ru.yandex.direct.useractionlog.writer;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.juggler.AsyncHttpJugglerClient;
import ru.yandex.direct.juggler.JugglerAsyncMultiSender;
import ru.yandex.direct.juggler.JugglerClient;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.JugglerSubject;
import ru.yandex.direct.useractionlog.LatestRecordTimeFetcher;
import ru.yandex.direct.utils.CommonUtils;
import ru.yandex.direct.utils.Transient;

import static ru.yandex.direct.useractionlog.db.DbConfigUtil.getShardReplicas;
import static ru.yandex.direct.useractionlog.db.DbConfigUtil.getWorkingReplica;

@ParametersAreNonnullByDefault
public class JugglerMonitor implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(JugglerMonitor.class);
    private static final int DEFAULT_POOL_SIZE = 1;

    private final JugglerSubject subject;
    private final LatestRecordTimeFetcher timeFetcher;
    private final Duration checkFrequency;
    private final Duration criticalDelay;
    private final Collection<String> jugglerGateways;

    private List<JugglerClient> clients;
    private JugglerAsyncMultiSender sender;
    private ScheduledExecutorService executor;
    private boolean needToShutdownExecutor;
    private boolean inited;

    public JugglerMonitor(
            DatabaseWrapperProvider wrapperProvider, DbConfigFactory configFactory,
            Duration checkFrequency,
            Duration criticalDelay, Collection<String> jugglerGateways,
            @Nullable ScheduledExecutorService executor
    ) {
        this.subject = new JugglerSubject("app.ual_writer.freshness");
        Collection<String> replicas = getShardReplicas(
                configFactory, SimpleDb.PPCHOUSE_PPC.toString() + ":shards:1");
        this.timeFetcher = new LatestRecordTimeFetcher(ignored -> getWorkingReplica(wrapperProvider, replicas));
        this.checkFrequency = checkFrequency;
        this.criticalDelay = criticalDelay;
        this.jugglerGateways = jugglerGateways;
        this.sender = null;
        this.executor = executor;
        this.needToShutdownExecutor = false;
        this.inited = false;
    }

    private ScheduledExecutorService createDefaultExecutor() {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("juggler-monitor-%02d").build();
        return Executors.newScheduledThreadPool(DEFAULT_POOL_SIZE, threadFactory);
    }

    public void init() {
        logger.info("init");
        if (!inited) {
            clients = new ArrayList<>();
            try {
                for (String jugglerGateway : jugglerGateways) {
                    clients.add(new AsyncHttpJugglerClient(jugglerGateway, true));
                }
                try (Transient<JugglerAsyncMultiSender> senderTransient =
                             new Transient<>(new JugglerAsyncMultiSender(clients))) {
                    if (executor == null) {
                        executor = createDefaultExecutor();
                        needToShutdownExecutor = true;
                    }
                    this.sender = senderTransient.pop();
                }
                executor.scheduleAtFixedRate(this::check, 0, checkFrequency.toMillis(), TimeUnit.MILLISECONDS);
                inited = true;
            } catch (Exception e) {
                clients.forEach(JugglerClient::close);
                throw e;
            }
        }
    }

    private void check() {
        logger.trace("running writer freshness check");
        try {
            Preconditions.checkState(inited, "Instance was not initialized");
            LocalDateTime now = ZonedDateTime.now(ZoneOffset.UTC).toLocalDateTime();
            LocalDate minRecordDate = now.minus(criticalDelay).toLocalDate();
            Map<String, LocalDateTime> latestRecordBySource = timeFetcher.getLatestRecordTime(minRecordDate);
            List<String> tardySources = new ArrayList<>();
            for (Map.Entry<String, LocalDateTime> entry : latestRecordBySource.entrySet()) {
                long delay;
                if (entry.getValue().equals(LocalDateTime.MIN)) {
                    delay = ChronoUnit.SECONDS.between(minRecordDate.atStartOfDay(), now);
                } else {
                    delay = Math.max(ChronoUnit.SECONDS.between(entry.getValue(), now), 0);
                }
                if (logger.isInfoEnabled()) {
                    logger.info("writer delay for {} is {} seconds",
                            entry.getKey(),
                            entry.getValue().equals(LocalDateTime.MIN) ? "more than " + delay : delay);
                }
                if (delay >= criticalDelay.getSeconds()) {
                    tardySources.add(entry.getKey());
                }
            }
            if (tardySources.isEmpty()) {
                sender.sendEvent(subject.makeEvent(JugglerStatus.OK, "OK"), checkFrequency);
            } else {
                sender.sendEvent(subject.makeEvent(
                        JugglerStatus.CRIT,
                        tardySources.size() + " source(s) are being late for more than " +
                                CommonUtils.formatApproxDuration(criticalDelay) + ": " +
                                String.join(", ", tardySources)),
                        checkFrequency);
            }
        } catch (RuntimeException e) {
            logger.error("Error while juggler check", e);
        }
    }

    @Override
    public void close() {
        logger.info("close");
        if (needToShutdownExecutor && !executor.isShutdown()) {
            executor.shutdown();
        }
        sender.close();
        clients.forEach(JugglerClient::close);
    }
}
