package ru.yandex.direct.useractionlog.writer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;

import ru.yandex.direct.useractionlog.db.ActionLogWriteRepository;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.useractionlog.schema.ActionLogSchema;

@ParametersAreNonnullByDefault
public class MemoryActionLogWriteRepository implements ActionLogWriteRepository {
    private final List<ActionLogRecord> records = new ArrayList<>();

    private static List<Object> actionLogRecordPrimaryKey(ActionLogRecord record) {
        Preconditions.checkState(Arrays.asList(ActionLogSchema.INDEX_COLUMNS).equals(Arrays.asList(
                ActionLogSchema.DATE,
                ActionLogSchema.PATH,
                ActionLogSchema.GTID,
                ActionLogSchema.QUERY_SERIAL,
                ActionLogSchema.ROW_SERIAL)));
        return Arrays.asList(record.getDateTime().toLocalDate(),
                record.getPath(),
                record.getGtid(),
                record.getQuerySerial(),
                record.getRowSerial());
    }

    public List<ActionLogRecord> getAllWrittenRecords() {
        return records;
    }

    public List<ActionLogRecord> getDistinctWrittenRecords() {
        return records
                .stream()
                .collect(Collectors.toMap(
                        MemoryActionLogWriteRepository::actionLogRecordPrimaryKey,
                        Function.identity(),
                        (ActionLogRecord o, ActionLogRecord n) ->
                                o.getRecordSource().compareTo(n.getRecordSource()) < 0 ? n : o))
                .values()
                .stream()
                .sorted(Comparator.comparing(ActionLogRecord::getDateTime)
                        .thenComparing(ActionLogRecord::getGtid)
                        .thenComparingInt(ActionLogRecord::getQuerySerial)
                        .thenComparingInt(ActionLogRecord::getRowSerial))
                .collect(Collectors.toList());
    }

    @Override
    public void insert(List<ActionLogRecord> records) {
        this.records.addAll(records);
    }
}
