package ru.yandex.direct.useractionlog.writer;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.concurrent.ThreadSafe;

import ru.yandex.direct.binlog.reader.StateBound;
import ru.yandex.direct.binlogclickhouse.BinlogStateDelayedSaver;
import ru.yandex.direct.binlogclickhouse.BinlogStateSaver;
import ru.yandex.direct.binlogclickhouse.StateBoundParallelConsumer;
import ru.yandex.direct.useractionlog.db.ActionLogWriteRepository;
import ru.yandex.direct.useractionlog.schema.ActionLogRecord;
import ru.yandex.direct.utils.Transient;

@ThreadSafe
public class StateBoundActionWriter implements Consumer<StateBound<List<ActionLogRecord>>>, AutoCloseable {
    private final BinlogStateDelayedSaver delayedSaver;
    private final StateBoundParallelConsumer<List<ActionLogRecord>> writer;

    public StateBoundActionWriter(ActionLogWriteRepository actionLogWriteRepository,
                                  BinlogStateSaver binlogStateSaver,
                                  Duration stateSaveDelay, int parallelism) {
        ActionWriter actionWriter = new ActionWriter(actionLogWriteRepository);
        // StateBoundParallelConsumer принимает на вход список всех возможных Consumer<StateBound<...>>
        // Это было удобно, когда каждый Consumer<> писал данные на один и тот же сервер.
        // Теперь WriteActionLogTable в связке с ShardReplicaChooser каждый раз пишет на новый сервер.
        try (Transient<BinlogStateDelayedSaver> transientDelayedSaver = new Transient<>(
                new BinlogStateDelayedSaver(binlogStateSaver, stateSaveDelay));
             Transient<StateBoundParallelConsumer<List<ActionLogRecord>>> transientParallelWriter =
                     new Transient<>(new StateBoundParallelConsumer<>(
                             Collections.nCopies(parallelism, actionWriter),
                             transientDelayedSaver.item, parallelism))) {
            writer = transientParallelWriter.pop();
            delayedSaver = transientDelayedSaver.pop();
        }
    }

    @Override
    public synchronized void accept(StateBound<List<ActionLogRecord>> actions) {
        writer.accept(actions);
    }

    @Override
    public void close() {
        try (BinlogStateDelayedSaver ignored = delayedSaver) {
            writer.close();
        }
    }
}
