package ru.yandex.direct.useractionlog.writer;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.useractionlog.Gtid;
import ru.yandex.direct.useractionlog.db.StateReaderWriter;
import ru.yandex.direct.useractionlog.db.UserActionLogStates;
import ru.yandex.direct.useractionlog.schema.UserActionLogStateType;

/**
 * Если встретилась ошибка вида
 * <code>The slave is connecting using CHANGE MASTER TO MASTER_AUTO_POSITION = 1, but the master has purged binary logs
 * containing GTIDs that the slave requires.</code>,
 * то возможны два варианта:
 * <ol>
 * <li>Писалка давно не запускалась и старые бинлоги удалены.</li>
 * <li>Как в DIRECT-77212: В gtidset неестественным образом появился некий новый uuid, и для него нет бинлога.</li>
 * </ol>
 * Этот класс позволяет добавить новый gtid или изменить существующий gtid в стейте.
 */
@ParametersAreNonnullByDefault
public class StateGtidSetEditor implements Runnable {
    private final String source;
    private final UserActionLogStateType type;
    private final Gtid gtidToAdd;
    private final StateReaderWriter stateReaderWriter;

    public StateGtidSetEditor(StateReaderWriter stateReaderWriter, String source, UserActionLogStateType type,
                              Gtid gtidToAdd) {
        this.source = source;
        this.type = type;
        this.gtidToAdd = gtidToAdd;
        this.stateReaderWriter = stateReaderWriter;
    }

    @Override
    public void run() {
        UserActionLogStates states = stateReaderWriter.read.getActualStates(source);
        MySQLBinlogState oldState;
        switch (type) {
            case LOG:
                oldState = states.getLog();
                break;
            case DICT:
                oldState = states.getDict();
                break;
            default:
                throw new IllegalStateException(type.toString());
        }
        Objects.requireNonNull(oldState, "Can't add anything to the empty state");
        List<Gtid> gtidSet = Gtid.fromGtidSet(oldState.getGtidSet());
        boolean noneFound = true;
        for (int idx = 0; idx < gtidSet.size(); ++idx) {
            if (gtidSet.get(idx).getUuid().equals(gtidToAdd.getUuid())) {
                gtidSet.set(idx, gtidToAdd);
                noneFound = false;
                break;
            }
        }
        if (noneFound) {
            gtidSet.add(gtidToAdd);
        }
        MySQLBinlogState newState = new MySQLBinlogState(oldState.getServerSchema(),
                gtidSet.stream()
                        .map(g -> String.format("%s:1-%s", g.getUuid(), g.getEventId()))
                        .collect(Collectors.joining(",")));
        switch (type) {
            case LOG:
                stateReaderWriter.write.saveLogState(source, newState);
                break;
            case DICT:
                stateReaderWriter.write.saveDictState(source, newState);
                break;
            default:
                throw new IllegalStateException(type.toString());
        }
    }
}
