package ru.yandex.direct.useractionlog.writer;

import java.sql.SQLException;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

import ru.yandex.direct.binlog.reader.BinlogSource;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.jetty.JettyConfig;
import ru.yandex.direct.common.jetty.JettyLauncher;
import ru.yandex.direct.common.liveresource.zookeeper.ZookeeperLiveResourceProvider;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.DirectConfigFactory;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.wrapper.DataSourceFactory;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.env.Environment;
import ru.yandex.direct.jcommander.Command;
import ru.yandex.direct.jcommander.ParserWithHelp;
import ru.yandex.direct.libs.curator.CuratorFrameworkProvider;
import ru.yandex.direct.libs.curator.lock.CuratorLock;
import ru.yandex.direct.libs.curator.lock.CuratorLockTimeoutException;
import ru.yandex.direct.liveresource.provider.ClasspathLiveResourceProvider;
import ru.yandex.direct.liveresource.provider.FileLiveResourceProvider;
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.metric.collector.MetricCollector;
import ru.yandex.direct.mysql.MySQLSimpleConnector;
import ru.yandex.direct.solomon.SolomonMonitoringServlet;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceLogger;
import ru.yandex.direct.tracing.TraceMdcAdapter;
import ru.yandex.direct.tracing.real.RealTrace;
import ru.yandex.direct.useractionlog.Gtid;
import ru.yandex.direct.useractionlog.TableNames;
import ru.yandex.direct.useractionlog.db.DbConfigClickHouseManager;
import ru.yandex.direct.useractionlog.db.DbSchemaCreator;
import ru.yandex.direct.useractionlog.db.StateReaderWriter;
import ru.yandex.direct.useractionlog.writer.initdictionaries.ExtendGtidSetCommand;
import ru.yandex.direct.useractionlog.writer.initdictionaries.InitDictionaries;
import ru.yandex.direct.useractionlog.writer.initdictionaries.InitDictionariesCommand;
import ru.yandex.direct.utils.Completer;
import ru.yandex.direct.utils.GracefulShutdownHook;
import ru.yandex.direct.version.DirectVersion;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@SuppressWarnings("squid:S106")  // System.out, System.err
public class UserActionLogTool {
    public static final String TRACE_SERVICE = "direct.user-action-log";
    private static final Duration JUGGLER_CHECK_FREQUENCY = Duration.ofMinutes(1);
    private static final Duration JUGGLER_CRITICAL_DELAY = Duration.ofMinutes(10);

    private static final Logger logger = LoggingInitializer.getLogger(UserActionLogTool.class);

    private UserActionLogTool() {
    }

    public static void main(String[] args) throws SQLException {
        CommonParams params = new CommonParams();
        Command command = ParserWithHelp.parseCommand(
                UserActionLogTool.class.getCanonicalName(),
                args,
                params,
                new DbCreationCommand(),
                new RunCommand(),
                new InitDictionariesCommand(),
                new ExtendGtidSetCommand()
        );

        if (!params.skipLogInit) {
            LoggingInitializer.initialize(params, ImmutableMap.of(
                    TraceMdcAdapter.SERVICE_KEY, TRACE_SERVICE,
                    TraceMdcAdapter.METHOD_KEY, command.getCommandName())
            );
        }

        var cfg = DirectConfigFactory.getConfig();

        var curator = new CuratorFrameworkProvider(
                cfg.getString("zookeeper.config_servers"),
                cfg.getString("zookeeper.lock_path")
        );

        var liveResourceFactory = new LiveResourceFactoryBean(List.of(
                new FileLiveResourceProvider(),
                new ClasspathLiveResourceProvider(),
                new ZookeeperLiveResourceProvider(curator)
        ));

        DbConfigFactory dbConfigFactory = new DbConfigFactory(
                liveResourceFactory.get(cfg.getString("db_config")).getContent());

        var dataSourceFactory = new DataSourceFactory(cfg);
        var databaseWrapperProvider =
                DatabaseWrapperProvider.newInstance(dataSourceFactory, dbConfigFactory, Environment.getCached());

        var dbConfigClickHouseManager = DbConfigClickHouseManager.create(dbConfigFactory,
                SimpleDb.PPCHOUSE_PPC.toString(), databaseWrapperProvider);

        var dslContextProvider = new DslContextProvider(databaseWrapperProvider);
        var ppcPropertiesSupport = new PpcPropertiesSupport(dslContextProvider);

        var traceScheduler = new ThreadPoolTaskScheduler();
        traceScheduler.setDaemon(true);
        traceScheduler.setThreadNamePrefix("TracingBackgroundT-");
        traceScheduler.setRemoveOnCancelPolicy(true);
        traceScheduler.initialize();

        var curatorLock = new CuratorFrameworkProvider(
                cfg.getString("zookeeper.servers"),
                cfg.getString("zookeeper.lock_path"));

        var zkLock = params.skipZkLock ? null : getCuratorLock(curatorLock);

        try {
            TraceLogger traceLogger = new TraceLogger((runnable, period, timeUnit) ->
                    traceScheduler.scheduleWithFixedDelay(runnable,
                            new Date(System.currentTimeMillis() + timeUnit.toMillis(period)),
                            timeUnit.toMillis(period)));

            Trace trace = RealTrace.builder()
                    .withService("direct.user-action-log")
                    .withMethod(Objects.requireNonNull(command.getCommandName()))
                    .build();
            Future traceRegistration = traceLogger.register(trace);
            Trace.push(trace);
            try {
                if (command instanceof DbCreationCommand) {
                    DbCreationCommand dbCreationCommand = (DbCreationCommand) command;
                    DbSchemaCreator dbSchemaCreator = new DbSchemaCreator(dbConfigClickHouseManager,
                            dbCreationCommand.distributedLabel, dbCreationCommand.mergeTreeLabel);
                    if (dbCreationCommand.script) {
                        System.out.println(dbSchemaCreator.bashScriptForCluster());
                    } else {
                        dbSchemaCreator.execute();
                    }
                } else if (command instanceof RunCommand) {
                    RunCommand runCommand = (RunCommand) command;

                    List<String> shardsDbnames = getShardsDbnames(dbConfigFactory, runCommand.shards);
                    Duration shutdownDuration = Duration.ofMinutes(5);
                    Completer.Builder completerBuilder = new Completer.Builder(shutdownDuration);
                    completerBuilder.submitVoid("writer", new UserActionLogWriter.Builder()
                            .withBinlogKeepAliveTimeout(runCommand.binlogKeepAliveTimeout)
                            .withBinlogSources(
                                    getBinlogSources(dbConfigFactory, shardsDbnames).collect(Collectors.toList()))
                            .withDbConfigFactory(dbConfigFactory)
                            .withBatchDuration(runCommand.logBatchDuration)
                            .withEventBatchSize(runCommand.logBatchSize)
                            .withRecordBatchSize(runCommand.recordBatchSize)
                            .withMysqlServerId(runCommand.mysqlServerId)
                            .withShardReplicaChooser(dbConfigClickHouseManager)
                            .withSkipErroneousEvents(runCommand.skipErroneousEvents)
                            .withDirectConfig(cfg)
                            .withUseTmpfs(runCommand.useTmpfs)
                            .withPpcPropertiesSupport(ppcPropertiesSupport)
                            .build());

                    @Nullable
                    JettyLauncher monitoringHttpServer = createMonitoringHttpServer(cfg);
                    registerMonitoringSensors();

                    try (GracefulShutdownHook ignored = new GracefulShutdownHook(shutdownDuration.plusSeconds(2));

                         MetricCollector metricCollector = makeAndRegisterMetricCollector(
                                 completerBuilder, cfg, databaseWrapperProvider, dbConfigFactory,
                                 shardsDbnames);
                         JugglerMonitor jugglerMonitor = makeAndRegisterJugglerMonitor(
                                 completerBuilder, cfg.getBranch("juggler"), databaseWrapperProvider, dbConfigFactory);
                         Completer completer = completerBuilder.build()) {
                        completer.waitAll();
                    } finally {
                        if (monitoringHttpServer != null) {
                            monitoringHttpServer.stop();
                        }
                    }
                } else if (command instanceof InitDictionariesCommand) {
                    InitDictionariesCommand initDictionariesCommand = (InitDictionariesCommand) command;
                    InitDictionaries initDictionaries = new InitDictionaries(dbConfigFactory,
                            databaseWrapperProvider,
                            dbConfigClickHouseManager,
                            initDictionariesCommand.clickhouseParallelWrites,
                            initDictionariesCommand.binlogKeepAliveTimeout,
                            initDictionariesCommand.mysqlServerId,
                            initDictionariesCommand.clickhouseChunkSize
                    );
                    initDictionaries.run();
                } else if (command instanceof ExtendGtidSetCommand) {
                    ExtendGtidSetCommand gtidSetCommand = (ExtendGtidSetCommand) command;
                    Gtid gtidToAdd = Gtid.fromGtidString(gtidSetCommand.addGtid);
                    StateReaderWriter stateTable = new StateReaderWriter(
                            dbConfigClickHouseManager::getForReading,
                            dbConfigClickHouseManager::getForWriting,
                            TableNames.USER_ACTION_LOG_STATE_TABLE);

                    new StateGtidSetEditor(stateTable, gtidSetCommand.shard, gtidSetCommand.stateType, gtidToAdd)
                            .run();
                }
            } finally {
                Trace.pop();
                traceRegistration.cancel(false);
                traceLogger.dump(trace);
            }
        } catch (InterruptedException e) {
            logger.info("Interrupted");
            Thread.currentThread().interrupt();
        } catch (RuntimeException e) {
            logger.error("Unexpected runtime exception", e);
            throw e;
        } finally {
            traceScheduler.shutdown();
            if (zkLock != null && zkLock.isAcquiredInThisProcess()) {
                try {
                    zkLock.release();
                } catch (Exception e) {
                    logger.error("Zookeeper lock release error", e);
                }
            }
        }
    }

    // Копия ru.yandex.direct.common.configuration.MetricsConfiguration.MetricsConfiguration
    private static void registerMonitoringSensors() {
        SolomonUtils.addJvmMetrics();

        SOLOMON_REGISTRY.lazyCounter("sensors_total", SOLOMON_REGISTRY::estimateCount);

        GaugeInt64 major = SOLOMON_REGISTRY.gaugeInt64("major_version");
        major.set(DirectVersion.getMajorVersion());

        GaugeInt64 minor = SOLOMON_REGISTRY.gaugeInt64("minor_version");
        minor.set(DirectVersion.getMinorVersion());
    }

    private static List<String> getShardsDbnames(DbConfigFactory dbConfigFactory, @Nullable List<String> shards) {
        if (shards == null) {
            return mapList(dbConfigFactory.getShardNumbers("ppc"), n -> "ppc:" + n);
        }
        return shards;
    }


    private static MetricCollector makeAndRegisterMetricCollector(
            Completer.Builder builder,
            DirectConfig directConfig,
            DatabaseWrapperProvider wrapperProvider,
            DbConfigFactory configFactory,
            Collection<String> shardNames
    ) {
        MetricCollector metricCollector = new MetricCollector(
                Collections.singletonList(new WriterMetricProvider(
                        directConfig, wrapperProvider, configFactory, shardNames)),
                null);
        builder.submitVoid("metricCollector", metricCollector::init);
        return metricCollector;
    }

    private static JugglerMonitor makeAndRegisterJugglerMonitor(
            Completer.Builder builder,
            DirectConfig jugglerConfig,
            DatabaseWrapperProvider wrapperProvider,
            DbConfigFactory configFactory
    ) {
        JugglerMonitor jugglerMonitor = new JugglerMonitor(
                wrapperProvider,
                configFactory,
                JUGGLER_CHECK_FREQUENCY,
                JUGGLER_CRITICAL_DELAY,
                Collections.singletonList(jugglerConfig.getString("eventsGateway")),
                null);
        builder.submitVoid("jugglerMonitor", jugglerMonitor::init);
        return jugglerMonitor;
    }

    private static Stream<BinlogSource> getBinlogSources(DbConfigFactory dbConfigFactory,
                                                         @Nonnull List<String> shards) {
        return shards
                .stream()
                .map(shard -> {
                    DbConfig dbConfig = dbConfigFactory.get(shard);
                    return new BinlogSource(
                            shard,
                            new MySQLSimpleConnector(
                                    dbConfig.getHosts().get(0),
                                    dbConfig.getPort(),
                                    dbConfig.getUser(),
                                    dbConfig.getPass()
                            )
                    );
                });
    }

    private static CuratorLock getCuratorLock(CuratorFrameworkProvider curator) {
        String lockName = "user-action-lock-writer_" + Environment.getCached() + "_lock";

        logger.info("Let's take zookeeper lock {}/{}", curator.getZkHosts(), curator.getLockPath().resolve(lockName));
        CuratorLock lock = null;
        while (true) {
            var timeout = Duration.ofMinutes(1);
            try {
                lock = curator.getLock(lockName,
                        timeout,
                        UserActionLogTool::zkLockLooseCallback);
                break;
            } catch (CuratorLockTimeoutException e) {
                logger.info("Can't get lock on timeout {}", timeout);
            }
        }
        logger.info("Taken zookeeper lock {}/{}", curator.getZkHosts(), curator.getLockPath().resolve(lockName));
        return lock;
    }

    private static JettyLauncher createMonitoringHttpServer(DirectConfig conf) {
        if (conf.findBoolean("monitoring_jetty.enable").orElse(false)) {
            JettyConfig jettyConfig = new JettyConfig(conf.getBranch("monitoring_jetty"));
            JettyLauncher jetty = JettyLauncher.server(jettyConfig)
                    .withServlet(new SolomonMonitoringServlet(), "/monitoring");
            jetty.start();
            return jetty;
        } else {
            return null;
        }
    }

    private static void zkLockLooseCallback() {
        logger.error("Lost zookeeper connection, let's stop program");
        System.exit(1);
    }
}
