package ru.yandex.direct.useractionlog.writer;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.Objects;
import java.util.concurrent.Semaphore;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;

import ru.yandex.direct.binlog.reader.BinlogSource;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.mysql.MySQLServerBuilder;
import ru.yandex.direct.useractionlog.TableNames;
import ru.yandex.direct.useractionlog.db.ReadWriteDictTable;
import ru.yandex.direct.useractionlog.db.ShardReplicaChooser;
import ru.yandex.direct.useractionlog.db.StateReaderWriter;
import ru.yandex.direct.useractionlog.db.WriteActionLogTable;
import ru.yandex.direct.useractionlog.dict.DictDefaults;
import ru.yandex.direct.useractionlog.schema.RecordSource;
import ru.yandex.direct.useractionlog.writer.generator.RowProcessingDefaults;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Completer;
import ru.yandex.direct.utils.io.TempDirectory;

/**
 * Запись пользовательских логов из binlog-соединений.
 */
@ParametersAreNonnullByDefault
public class UserActionLogWriter implements Checked.CheckedRunnable<InterruptedException> {
    private static final Duration WAIT_TERMINATION = Duration.ofMinutes(1);

    private final Collection<BinlogSource> binlogSources;
    @Nullable
    private final Integer mysqlServerId;
    private final int eventBatchSize;
    private final int recordBatchSize;
    private final boolean useTmpfs;
    private final Duration logBatchDuration;
    private final boolean skipErroneousEvents;
    private final DbConfigFactory dbConfigFactory;

    private final WriteActionLogTable writeActionLogTable;
    private final ReadWriteDictTable dictTable;
    private final StateReaderWriter stateTable;  // TODO rename
    private final DirectConfig directConfig;
    private final Duration binlogKeepAliveTimeout;
    private final RecordSource recordSource;
    private final Semaphore binlogStateFetchingSemaphore;
    private final Semaphore schemaReplicaMysqlBuilderSemaphore;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    // Конструктор с большим количеством аргументом. Прикрывается билдером.
    @SuppressWarnings({"squid:S00107", "checkstyle:parameternumber"})
    private UserActionLogWriter(
            boolean skipErroneousEvents,
            boolean useTmpfs,
            Collection<BinlogSource> binlogSources,
            DbConfigFactory dbConfigFactory,
            Duration binlogKeepAliveTimeout,
            Duration logBatchDuration,
            int eventBatchSize,
            int recordBatchSize,
            ShardReplicaChooser shardReplicaChooser,
            @Nullable Integer mysqlServerId,
            DirectConfig directConfig,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.skipErroneousEvents = skipErroneousEvents;
        this.useTmpfs = useTmpfs;
        this.binlogSources = binlogSources;
        this.mysqlServerId = mysqlServerId;
        this.eventBatchSize = eventBatchSize;
        this.recordBatchSize = recordBatchSize;
        this.logBatchDuration = logBatchDuration;
        this.dbConfigFactory = dbConfigFactory;
        this.binlogKeepAliveTimeout = binlogKeepAliveTimeout;

        this.writeActionLogTable = new WriteActionLogTable(shardReplicaChooser::getForWriting,
                TableNames.WRITE_USER_ACTION_LOG_TABLE);
        this.dictTable = new ReadWriteDictTable(
                shardReplicaChooser::getForReading,
                shardReplicaChooser::getForWriting,
                TableNames.DICT_TABLE);
        this.stateTable = new StateReaderWriter(
                shardReplicaChooser::getForReading,
                shardReplicaChooser::getForWriting,
                TableNames.USER_ACTION_LOG_STATE_TABLE);
        this.directConfig = directConfig;
        this.recordSource = RecordSource.makeDaemonRecordSource();
        this.binlogStateFetchingSemaphore = new Semaphore(4);
        this.schemaReplicaMysqlBuilderSemaphore = new Semaphore(4);
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    public void run() throws InterruptedException {
        MySQLServerBuilder schemaReplicaMysqlBuilder = new MySQLServerBuilder()
                .setMysqldBinary(directConfig.getStringList("mysqld_binary_path"))
                .withTempDirectoryProvider(this::createMysqlTempDirectory)
                .setGracefulStopTimeout(Duration.ZERO)
                .addExtraArgs(Collections.singletonList("--skip-innodb-use-native-aio"))
                .withNoSync(true);

        Completer.Builder completerBuilder = new Completer.Builder(WAIT_TERMINATION);
        for (BinlogSource binlogSource : binlogSources) {
            completerBuilder.submitVoid("UserActionLogWriter-" + binlogSource.getName(),
                    () -> new ActionProcessor.Builder()
                            .withBatchDuration(logBatchDuration)
                            .withBinlogKeepAliveTimeout(binlogKeepAliveTimeout)
                            .withBinlogStateFetchingSemaphore(binlogStateFetchingSemaphore)
                            .withDirectConfig(directConfig)
                            .withDbConfig(dbConfigFactory.get(binlogSource.getName()))
                            .withDictRepository(
                                    DictDefaults.makeReadWriteDictRepository(binlogSource.getName(), dictTable))
                            .withEventBatchSize(eventBatchSize)
                            .withPpcPropertiesSupport(ppcPropertiesSupport)
                            .withRecordBatchSize(recordBatchSize)
                            .withInitialServerId(mysqlServerId)
                            .withReadWriteStateTable(stateTable)
                            .withRowProcessingStrategy(
                                    RowProcessingDefaults.defaultRowToActionLog(recordSource))
                            .withSchemaReplicaMysqlBuilder(schemaReplicaMysqlBuilder.copy())
                            .withSchemaReplicaMysqlSemaphore(schemaReplicaMysqlBuilderSemaphore)
                            .withSkipErroneousEvents(skipErroneousEvents)
                            .withUntilGtidSet(null)
                            .withWriteActionLogTable(writeActionLogTable)
                            .build()
                            .run());
        }
        try (Completer completer = completerBuilder.build()) {
            completer.waitAll();
        }
    }

    public static class Builder {
        private Collection<BinlogSource> binlogSources;
        private ShardReplicaChooser shardReplicaChooser;
        private Integer mysqlServerId;
        private boolean mysqlServerIdSet;
        private Integer eventBatchSize = ActionProcessor.DEFAULT_EVENT_BATCH_SIZE;
        private int recordBatchSize = ActionProcessor.DEFAULT_RECORD_BATCH_SIZE;
        private Duration batchDuration;
        private Boolean skipErroneousEvents;
        private Boolean useTmpfs;
        private DbConfigFactory dbConfigFactory;
        private Duration binlogKeepAliveTimeout;
        private DirectConfig directConfig;
        private PpcPropertiesSupport ppcPropertiesSupport;

        public Builder withBinlogSources(Collection<BinlogSource> binlogSources) {
            this.binlogSources = binlogSources;
            return this;
        }

        public Builder withShardReplicaChooser(ShardReplicaChooser shardReplicaChooser) {
            this.shardReplicaChooser = shardReplicaChooser;
            return this;
        }

        public Builder withMysqlServerId(Integer mysqlServerId) {
            mysqlServerIdSet = true;
            this.mysqlServerId = mysqlServerId;
            return this;
        }

        public Builder withEventBatchSize(int logBatchSize) {
            this.eventBatchSize = logBatchSize;
            return this;
        }

        public Builder withRecordBatchSize(int recordBatchSize) {
            this.recordBatchSize = recordBatchSize;
            return this;
        }

        public Builder withBatchDuration(Duration logBatchDuration) {
            this.batchDuration = logBatchDuration;
            return this;
        }

        public Builder withBinlogKeepAliveTimeout(Duration binlogKeepAliveTimeout) {
            this.binlogKeepAliveTimeout = binlogKeepAliveTimeout;
            return this;
        }

        public Builder withSkipErroneousEvents(boolean skipErroneousEvents) {
            this.skipErroneousEvents = skipErroneousEvents;
            return this;
        }

        public Builder withUseTmpfs(boolean useTmpfs) {
            this.useTmpfs = useTmpfs;
            return this;
        }

        public Builder withDbConfigFactory(DbConfigFactory dbConfigFactory) {
            this.dbConfigFactory = dbConfigFactory;
            return this;
        }

        public Builder withDirectConfig(DirectConfig cfg) {
            this.directConfig = cfg;
            return this;
        }

        public Builder withPpcPropertiesSupport(PpcPropertiesSupport ppcPropertiesSupport) {
            this.ppcPropertiesSupport = ppcPropertiesSupport;
            return this;
        }

        public UserActionLogWriter build() {
            Objects.requireNonNull(binlogSources, "Forgotten binlogSources");
            Objects.requireNonNull(dbConfigFactory, "Forgotten dbConfigFactory");
            Objects.requireNonNull(batchDuration, "Forgotten batchDuration");
            Objects.requireNonNull(eventBatchSize, "Forgotten eventBatchSize");
            Objects.requireNonNull(shardReplicaChooser, "Forgotten shardReplicaChooser");
            Objects.requireNonNull(skipErroneousEvents, "Forgotten skipErroneousEvents");
            Objects.requireNonNull(useTmpfs, "Forgotten useTmpfs");
            Objects.requireNonNull(directConfig, "Forgotten directConfig");
            Preconditions.checkState(mysqlServerIdSet, "Forgotten mysqlServerId");

            return new UserActionLogWriter(
                    skipErroneousEvents,
                    useTmpfs,
                    binlogSources,
                    dbConfigFactory,
                    binlogKeepAliveTimeout,
                    batchDuration,
                    eventBatchSize,
                    recordBatchSize,
                    shardReplicaChooser,
                    mysqlServerId,
                    directConfig,
                    ppcPropertiesSupport);
        }
    }

    private TempDirectory createMysqlTempDirectory(String prefix) {
        Path preferredPath = Paths.get("/dev/shm");
        if (useTmpfs && preferredPath.toFile().isDirectory() && preferredPath.toFile().canWrite()) {
            // Если у нас есть права записи в /dev/shm, то создаём директорию там
            return new TempDirectory(preferredPath, "mysql-ualw-" + prefix);
        } else {
            return new TempDirectory("mysql-ualw-" + prefix);
        }
    }
}
