package ru.yandex.direct.useractionlog.writer.initdictionaries;

import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import org.jooq.Record;
import org.jooq.SelectWhereStep;
import org.jooq.TableField;

/**
 * Чтение данных из большого SELECT-запроса пачками. Каждая новая пачка получается на основе исходного SQL-запроса с
 * добавлением <code>WHERE primaryKey >= lastPrimaryKeyValue ORDER BY primaryKey LIMIT chunk</code>
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
class JooqChunkReader<R extends Record, K> implements Iterator<List<R>> {
    private final int chunkSize;
    private final TableField<?, K> primaryKeyTableField;
    private final Supplier<SelectWhereStep<R>> initialSelector;
    private boolean finalChunkWasRead;

    @Nullable
    private List<R> chunk = null;

    @Nullable
    private K primaryKeyLastValue = null;

    /**
     * @param primaryKeyTableField Колонка с первичным ключом. Должна быть в запросе.
     * @param initialSelector      Функция, которая каждый раз возвращает одинаковый jooq-запрос (но не один и тот
     *                             же объект)
     * @param chunkSize            Максимальное количество строк, которое будет прочитано из результатов запроса.
     */
    JooqChunkReader(TableField<?, K> primaryKeyTableField, Supplier<SelectWhereStep<R>> initialSelector,
                    @Nullable K primaryKeyStartValue, int chunkSize) {
        this.primaryKeyTableField = primaryKeyTableField;
        this.initialSelector = initialSelector;
        this.chunkSize = chunkSize;
        this.primaryKeyLastValue = primaryKeyStartValue;
    }

    @Override
    public boolean hasNext() {
        fetchChunk();
        return !finalChunkWasRead || chunk != null;
    }

    @Override
    public List<R> next() {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        List<R> result = chunk;
        chunk = null;
        return result;
    }

    private void fetchChunk() {
        if (chunk != null || finalChunkWasRead) {
            return;
        }
        SelectWhereStep<R> selector = initialSelector.get();
        if (primaryKeyLastValue != null) {
            selector.where(primaryKeyTableField.gt(primaryKeyLastValue));
        }
        selector.orderBy(primaryKeyTableField).limit(chunkSize);
        chunk = selector.fetch();
        if (chunk.size() < chunkSize) {
            finalChunkWasRead = true;
        } else {
            primaryKeyLastValue = chunk.get(chunk.size() - 1).get(primaryKeyTableField);
        }
    }
}
