package ru.yandex.direct.useractionlog.writer.initdictionaries;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.LinkedBlockingQueue;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.jooq.DSLContext;
import org.jooq.EnumType;
import org.jooq.Record;
import org.jooq.SelectField;
import org.jooq.TableField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.reader.EnrichedInsertRow;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.mysql.MySQLSimpleRow;
import ru.yandex.direct.mysql.schema.ColumnSchema;
import ru.yandex.direct.mysql.schema.TableSchema;
import ru.yandex.direct.mysql.schema.TableType;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.useractionlog.Gtid;

/**
 * Скачивает таблицу последовательными SELECT-запросами и представляет данные
 * в виде последовательных {@link EnrichedInsertRow}.
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
@SuppressWarnings("unchecked")
        // JooqChunkReader требует точные типы. Их сложно предоставить. Можно избавиться от этого, но потом.
class SqlDictFetcher extends Thread {
    // Фиктивный нулевой server uuid используется из-за того, что при переносе из времменного словаря
    // в постоянный мы сравниваем gtid (чего делать в общем случае нельзя, т.к. на server uuid
    // нет порядка), а нулевой uuid лексикографически меньше любого другого, поэтому, если у строки
    // было изменение, то в основной словарь уедет именно оно.
    public static final String FETCHER_SERVER_UUID = "00000000-0000-0000-0000-000000000000";

    private static final Logger logger = LoggerFactory.getLogger(SqlDictFetcher.class);
    private static final TableSchema DUMMY_TABLE_SCHEMA = new TableSchema(
            "ignored", TableType.TABLE, "ignored",
            Collections.emptyList(), Collections.emptyList(), Collections.emptyList());
    private Gtid gtid;
    private final String dbName;
    private final String tableName;
    private final ImmutableSet<TableField> fields;
    private final ImmutableMap<String, Integer> nameMap;
    private final ImmutableList<ColumnSchema> columnSchemas;
    private final JooqChunkReader<Record, ?> jooqChunkReader;
    private final List<TableField> primaryKey;
    private final BlockingQueue<CompletableFuture<Chunk>> resultQueue;
    private int rowSerial;

    /**
     * @param dslContext Объект, позволяющий получить соединение с БД
     * @param eventId    eventId, с которым будет вставлен первый {@link EnrichedInsertRow}.
     * @param chunkSize  Максимальный размер пачки кортежей, которая может быть считана из таблицы MySQL.
     */
    SqlDictFetcher(ImmutableSet<TableField> fields, DSLContext dslContext, long eventId, @Nullable Long startFromPk,
                   int chunkSize) {
        this.fields = fields;
        this.gtid = new Gtid(FETCHER_SERVER_UUID, eventId);

        primaryKey = fields.iterator().next().getTable().getPrimaryKey().getFields();
        if (primaryKey.size() != 1) {
            throw new UnsupportedOperationException("Currently supported only tables with one-column primary key");
        }
        this.dbName = primaryKey.get(0).getTable().getSchema().getName();
        this.tableName = primaryKey.get(0).getTable().getName();

        int nameMapCounter = 0;
        ImmutableMap.Builder<String, Integer> nameMapBuilder = ImmutableMap.builder();
        ImmutableList.Builder<ColumnSchema> columnSchemaBuilder = ImmutableList.builder();
        for (TableField field : fields) {
            nameMapBuilder.put(field.getName(), nameMapCounter++);
            columnSchemaBuilder.add(new ColumnSchema(field.getName(),
                    field.getDataType().getTypeName(),
                    "ignored",
                    null,
                    field.getDataType().nullable()));
        }
        nameMap = nameMapBuilder.build();
        columnSchemas = columnSchemaBuilder.build();

        this.jooqChunkReader = new JooqChunkReader(
                primaryKey.get(0),
                () -> dslContext
                        .select(fields.toArray(new SelectField[fields.size()]))
                        .from(tableName),
                startFromPk,
                chunkSize);

        // Между SELECT-запросами позволительна задержка, в то же время все данные из таблицы не поместятся
        // в оперативку, поэтому очередь ограничена.
        this.resultQueue = new LinkedBlockingQueue<>(3);
        this.rowSerial = 0;
    }

    /**
     * Очередь, в которую будут складываться пачки кортежей. {@code null} свидетельствует о завершении работы.
     * Может передать {@code RuntimeException}, если таковой возник в ходе выполнения работы, после которого не следует
     * ждать сообщений.
     */
    BlockingQueue<CompletableFuture<Chunk>> getResultQueue() {
        return resultQueue;
    }

    @Override
    public void run() {
        try {
            runWithInterrupts();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.info("SQL dict fetcher thread interrupted");
        }
    }

    private void runWithInterrupts() throws InterruptedException {
        try {
            while (jooqChunkReader.hasNext()) {
                List<Record> records = jooqChunkReader.next();
                logger.debug("Fetched {} rows from table {}", records.size(), tableName);
                List<EnrichedRow> rows = new ArrayList<>();
                for (Record record : records) {
                    Serializable[] values = new Serializable[fields.size()];
                    int i = 0;
                    for (TableField field : fields) {
                        Object source = record.get(field);
                        Serializable newValue;
                        if (source instanceof EnumType) {
                            newValue = ((EnumType) source).getLiteral();
                        } else {
                            newValue = (Serializable) source;
                        }
                        values[i++] = newValue;
                    }
                    EnrichedInsertRow enrichedInsertRow = new EnrichedInsertRow(gtid.toString(),
                            "/* ignored query */",
                            DirectTraceInfo.empty(),
                            0,
                            0,
                            rowSerial++,
                            LocalDateTime.now(ZoneId.of("UTC")),
                            dbName,
                            tableName,
                            // схема таблицы здесь не нужна
                            DUMMY_TABLE_SCHEMA,
                            new MySQLSimpleRowIndexed(nameMap, new MySQLSimpleRow(columnSchemas, values)));
                    rows.add(enrichedInsertRow);
                    gtid = new Gtid(gtid.getUuid(), gtid.getEventId() + 1);
                }
                CompletableFuture<Chunk> future = new CompletableFuture<>();
                if (records.isEmpty()) {
                    future.complete(new Chunk(rows, null, gtid.getEventId()));
                } else {
                    future.complete(new Chunk(
                            rows, (long) records.get(records.size() - 1).get(primaryKey.get(0)), gtid.getEventId()));
                }
                resultQueue.put(future);
            }
            CompletableFuture<Chunk> future = new CompletableFuture<>();
            future.complete(new Chunk(Collections.emptyList(), null, gtid.getEventId()));
            resultQueue.put(future);
        } catch (RuntimeException e) {
            CompletableFuture<Chunk> future = new CompletableFuture<>();
            future.completeExceptionally(e);
            resultQueue.put(future);
        }
    }

    class Chunk {
        /**
         * Список кортежей, полученных SELECT-запросом
         */
        final List<EnrichedRow> rows;

        /**
         * primary key из последнего кортежа
         */
        final Long lastReadPk;

        /**
         * eventId следующий за последним вставленным
         */
        final long lastEventId;

        Chunk(List<EnrichedRow> rows, @Nullable Long lastReadPk, long lastEventId) {
            this.rows = rows;
            this.lastReadPk = lastReadPk;
            this.lastEventId = lastEventId;
        }
    }
}
