package ru.yandex.direct.useractionlog.writer.initdictionaries;

import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.binlog.reader.EnrichedDeleteRow;
import ru.yandex.direct.binlog.reader.EnrichedInsertRow;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;

import static ru.yandex.direct.dbschema.ppc.Ppc.PPC;

class Util {
    static final ImmutableMap<String, Function<MySQLSimpleRowIndexed, List<String>>> PRIMARY_KEY_MAP =
            makePrimaryKeyMap(
                    PPC.BANNERS, PPC.CAMP_OPTIONS, PPC.CAMPAIGNS, PPC.CLIENTS, PPC.CLIENTS_OPTIONS,
                    PPC.DEMOGRAPHY_MULTIPLIER_VALUES, PPC.GEO_MULTIPLIER_VALUES, PPC.HIERARCHICAL_MULTIPLIERS,
                    PPC.PHRASES, PPC.RETARGETING_CONDITIONS, PPC.RETARGETING_MULTIPLIER_VALUES);

    private Util() {
        throw new UnsupportedOperationException("static members only");
    }

    @Nonnull
    private static MySQLSimpleRowIndexed dataForGettingId(EnrichedRow row) {
        if (row instanceof EnrichedUpdateRow) {
            return ((EnrichedUpdateRow) row).getFields().getBefore();
        } else if (row instanceof EnrichedInsertRow) {
            return ((EnrichedInsertRow) row).getFields();
        } else if (row instanceof EnrichedDeleteRow) {
            return ((EnrichedDeleteRow) row).getFields();
        } else {
            throw new UnsupportedOperationException("Unknown type of row " + row);
        }
    }

    private static ImmutableMap<String, Function<MySQLSimpleRowIndexed, List<String>>> makePrimaryKeyMap(
            Table... tables) {
        ImmutableMap.Builder<String, Function<MySQLSimpleRowIndexed, List<String>>> builder =
                new ImmutableMap.Builder<>();
        for (Table table : tables) {
            builder.put(table.getName(), row -> {
                ImmutableList.Builder<String> listBuilder = new ImmutableList.Builder<>();
                listBuilder.add(table.getName());
                for (TableField field : table.getPrimaryKey().getFieldsArray()) {
                    listBuilder.add(row.getByNameNullable(field.getName()).getValueAsString());
                }
                return listBuilder.build();
            });
        }
        return builder.build();
    }

    static List<String> getPrimaryKey(EnrichedRow row) {
        MySQLSimpleRowIndexed data = dataForGettingId(row);
        String tableName = row.getTableName();
        if (!PRIMARY_KEY_MAP.containsKey(tableName)) {
            throw new IllegalArgumentException("unsupported table " + tableName);
        }
        return PRIMARY_KEY_MAP.get(tableName).apply(data);
    }

}
