package ru.yandex.direct.apps.yt.benchmark;

import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.graphite.GraphiteMetricsBuffer;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.utils.io.RuntimeIoException;
import ru.yandex.monlib.metrics.labels.Labels;

public class Benchmark implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(Benchmark.class);

    private final DirectConfig directConfig;
    private final BenchmarkParams params;

    private final AsyncHttpClient asyncHttpClient;
    private final SolomonPushClient solomonPushClient;

    private final List<YtClientApi> clients;
    private final List<Query> queries;

    public Benchmark(DirectConfig directConfig, BenchmarkParams params) {
        this.directConfig = directConfig;
        this.params = params;

        YtConfig cfg = params.cluster.config();

        asyncHttpClient = new DefaultAsyncHttpClient();
        solomonPushClient = createSolomonPushClient(directConfig, asyncHttpClient);

        Map<YtClient, YtClientApi> clientsMap = ImmutableMap.of(
                YtClient.KOSHER, new YtClientKosherImpl(cfg),
                YtClient.RPC_CLIENT, new YtClientRpcImpl(cfg, false),
                YtClient.RPC_CLIENT_ASYNC, new YtClientRpcImpl(cfg, true)
        );
        this.clients = params.clients.isEmpty()
                ? new ArrayList<>(clientsMap.values())
                : StreamEx.of(params.clients).map(clientsMap::get).toList();

        QueriesRepository queriesRepository = new QueriesRepository(params);
        queries = queriesRepository.getQueries();
    }

    public void doIt() {
        logger.info(String.format("Start benchmark: cluster %s, %d queries, %d clients," +
                        " warmup iterations %d, benchmark iterations %d",
                params.cluster, queries.size(), clients.size(), params.warmup, params.iterations));
        for (Query query : queries) {
            if (logger.isTraceEnabled()) {
                logger.trace("Query: " + query.query());
            }
            for (YtClientApi client : clients) {
                warmup(client, query);
                benchmark(client, query);
            }
        }
    }

    public void warmup(YtClientApi client, Query query) {
        String name = client.name() + "@" + params.cluster.name() + "/" + query.name();
        logger.debug("start warmup for " + name);
        for (int i = 0; i < params.warmup; i++) {
            try {
                client.selectRows(query.query());
            } catch (RuntimeException ex) {
                logger.warn("warmup error for " + name, ex);
            }
        }
    }

    public void benchmark(YtClientApi client, Query query) {
        GraphiteMetricsBuffer metrics = new GraphiteMetricsBuffer(params.solomonFlow);
        String name = client.name() + "@" + params.cluster.name() + "/" + query.name();
        logger.debug("start measure for " + name);

        Stats stats = measure(name, params, client, query);
        logger.info("Stats for " + name + ": " + stats.format());

        if (params.sendToSolomon) {
            var registry = SolomonUtils.newPushRegistry(Labels.of(
                    "flow", params.solomonFlow,
                    "client", client.name(),
                    "cluster", params.cluster.name().toLowerCase(),
                    "query", query.name()
            ));
            stats.writeMetrics(registry);
        }
    }

    private static Stats measure(String name, BenchmarkParams params, YtClientApi client, Query query) {
        Stats stats = new Stats();
        for (int i = 0; i < params.iterations; i++) {
            long startTime = System.nanoTime();
            try {
                int ret = client.selectRows(query.query());
                Duration ela = Duration.ofNanos(System.nanoTime() - startTime);
                stats.addSuccessTime(ela.toMillis());
            } catch (RuntimeException ex) {
                stats.addError(ex);
                logger.warn("measure error for " + name, ex);
            }
        }

        return stats;
    }

    @Override
    public void close() {
        for (YtClientApi client : clients) {
            client.close();
        }
        try {
            asyncHttpClient.close();
        } catch (IOException ex) {
            throw new RuntimeIoException(ex);
        }
    }

    private SolomonPushClient createSolomonPushClient(DirectConfig directConfig, AsyncHttpClient asyncHttpClient) {
        DirectConfig cfg = directConfig.getBranch("solomon.push");
        var parallelFetcherFactory = new ParallelFetcherFactory(
                this.asyncHttpClient,
                new FetcherSettings()
                        .withGlobalTimeout(cfg.findDuration("global_timeout").orElse(Duration.ofSeconds(3)))
                        .withRequestTimeout(cfg.findDuration("request_timeout").orElse(Duration.ofSeconds(3)))
                        .withRequestRetries(cfg.findInt("retries").orElse(2))
                        .withSoftTimeout(cfg.findDuration("soft_timeout").orElse(Duration.ofSeconds(1)))
        );
        return new SolomonPushClient(cfg.getString("url"), parallelFetcherFactory);
    }
}
