package ru.yandex.autotests.direct.api.adgroups.add;

import java.util.Arrays;

import com.yandex.direct.api.v5.adgroups.SmartAdGroupAdd;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.adgroups.AdGroupsFeatures;
import ru.yandex.autotests.directapi.apiclient.config.Semaphore;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.darkside.datacontainers.jsonrpc.fake.GroupFakeInfo;
import ru.yandex.autotests.directapi.darkside.model.Status;
import ru.yandex.autotests.directapi.model.api5.adgroups.AdGroupAddItemMap;
import ru.yandex.autotests.directapi.model.api5.adgroups.AddRequestMap;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.rules.Trashman;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.autotests.irt.testutils.allure.TestSteps;
import ru.yandex.direct.core.entity.adgroup.service.validation.types.PerformanceAdGroupValidation;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.info.FeedInfo;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.hazelcast.SemaphoreRule;

import static org.hamcrest.Matchers.equalTo;

//import ru.yandex.autotests.direct.utils.tags.TagDictionary;

/**
 * Author pavryabov
 * Date 14.07.14
 * https://st.yandex-team.ru/TESTIRT-2022
 */
@Aqua.Test
@Features(AdGroupsFeatures.ADD)
public class AddTest {
    private static final long NONEXISTENT_FEED_ID = 39234L;
    private static final String FEED_TITLE = "title";
    private static final String FEED_BODY = "body";

    private static ClientInfo client;

    @ClassRule
    public static final ApiSteps api = new ApiSteps();

    @ClassRule
    public static final SemaphoreRule semaphore = Semaphore.getSemaphore();

    @Rule
    public Trashman trashman = new Trashman(api);

    private static Long campaignId;
    private static Long performanceCampaignId;
    private static Long feedId;

    @BeforeClass
    public static void initTest() {
        client = api.createClient();
        api.as(client);

        campaignId = api.userSteps().campaignSteps().addDefaultTextCampaign(client);
        performanceCampaignId = api.userSteps().campaignSteps().addDefaultSmartCampaign(client);
        feedId = api.userSteps().feedSteps().addDefaultFeed(client).getFeedId();
    }

    @Before
    public void clearSpentUnits() {
        api.userSteps().clientFakeSteps().fakeClearClientSpentUnits(client.getLogin());
    }

    @Test
    public void addWithValidParams() {
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.success()
        );
    }

    @Test
    public void addTwoGroupsWithSameName() {
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.success(), ExpectedResult.success()
        );
    }

    @Test
    public void add100Groups() {
        AdGroupAddItemMap[] adGroupAddItemMaps = new AdGroupAddItemMap[100];
        Arrays.fill(adGroupAddItemMaps, new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId));
        ExpectedResult[] expectedResults = new ExpectedResult[100];
        Arrays.fill(expectedResults, ExpectedResult.success());
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(adGroupAddItemMaps),
                expectedResults
        );
    }

    @Test
    public void addTwoInvalidGroups() {
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId + 10000),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId + 10000)
                ),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND))
        );
    }

    @Test
    public void addWithInvalidAndValidGroups() {
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId + 10000),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.CAMPAIGN_NOT_FOUND)),
                ExpectedResult.success()
        );
    }

    @Test
    public void addWithTwoInvalidFieldsInOneGroup() {
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap()
                                .withCampaignId(campaignId)
                                .withRegionIds(AdGroupAddItemMap.DEFAULT_REGION_ID)
                                .withName(" ")
                                .withNegativeKeywords("test]")
                ),
                ExpectedResult.errors(
                        new Notification(5002, Api5ErrorDetails.INCORRECT_USE_OF_BRACKETS_IN_MINUS_PHRASE, "test]"),
                        new Notification(5004, Api5ErrorDetails.GROUP_NAME_CANNOT_BE_EMPTY_STRING))
        );
    }

    @Test
    public void add1001GroupsInOneCampaign() {
        //DIRECT-34455
        Long campaignIdFor1000Groups = api.userSteps().campaignSteps().addDefaultTextCampaign(client);
        AdGroupAddItemMap[] adGroupAddItemMaps = new AdGroupAddItemMap[100];
        Arrays.fill(adGroupAddItemMaps, new AdGroupAddItemMap().defaultAdGroupAddItem(campaignIdFor1000Groups));
        for (int i = 0; i < 10; i++) {
            api.userSteps().adGroupsSteps().adGroupsAdd(new AddRequestMap().withAdGroups(adGroupAddItemMaps));
        }
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignIdFor1000Groups)),
                ExpectedResult.errors(
                        new Notification(7001, Api5ErrorDetailsJava.MAXIMUM_GROUPS_IN_CAMPAIGN_REACHED))
        );
    }

    @Test
    public void addWithDifferentSetsOfFields() {
        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                                .withNegativeKeywords(AdGroupAddItemMap.DEFAULT_NEGATIVE_KEYWORD),
                        new AdGroupAddItemMap().defaultAdGroupAddItem(campaignId)
                ),
                ExpectedResult.success(),
                ExpectedResult.success()
        );
    }

    @Test
    public void addPerformanceWithValidParams() {
        var smartAdGroup = new SmartAdGroupAdd()
                .withFeedId(feedId)
                .withAdTitleSource(FEED_TITLE)
                .withAdBodySource(FEED_BODY);
        var adGroupAddItemMap = new AdGroupAddItemMap()
                .defaultAdGroupAddItem(performanceCampaignId)
                .withPerformanceAdGroup(smartAdGroup);

        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(adGroupAddItemMap),
                ExpectedResult.success()
        );
    }

    @Test
    public void addPerformanceWithNonExistentFeed() {
        var smartAdGroup = new SmartAdGroupAdd().withFeedId(NONEXISTENT_FEED_ID);
        var adGroupAddItemMap = new AdGroupAddItemMap()
                .defaultAdGroupAddItem(performanceCampaignId)
                .withPerformanceAdGroup(smartAdGroup);

        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(adGroupAddItemMap),
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.FEED_NOT_EXIST_AD_GROUP,
                        NONEXISTENT_FEED_ID))
        );
    }

    @Test
    public void addPerformanceWithInvalidFeedTitle() {
        var smartAdGroup = new SmartAdGroupAdd()
                .withFeedId(feedId)
                .withAdTitleSource("a".repeat(PerformanceAdGroupValidation.MAX_NAME_LENGTH + 1))
                .withAdBodySource(FEED_BODY);
        var adGroupAddItemMap = new AdGroupAddItemMap()
                .defaultAdGroupAddItem(performanceCampaignId)
                .withPerformanceAdGroup(smartAdGroup);

        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(adGroupAddItemMap),
                ExpectedResult.errors(new Notification(5001, Api5ErrorDetails.FEED_NAME_FIELD_IS_TOO_LONG,
                        PerformanceAdGroupValidation.MAX_NAME_LENGTH))
        );
    }

    @Test
    public void addPerformanceWithInvalidFeedBody() {
        var smartAdGroup = new SmartAdGroupAdd()
                .withFeedId(feedId)
                .withAdTitleSource(FEED_TITLE)
                .withAdBodySource("a".repeat(PerformanceAdGroupValidation.MAX_BODY_LENGTH + 1));
        var adGroupAddItemMap = new AdGroupAddItemMap()
                .defaultAdGroupAddItem(performanceCampaignId)
                .withPerformanceAdGroup(smartAdGroup);

        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(adGroupAddItemMap),
                ExpectedResult.errors(new Notification(5001, Api5ErrorDetails.FEED_BODY_FIELD_IS_TOO_LONG,
                        PerformanceAdGroupValidation.MAX_BODY_LENGTH))
        );
    }

    @Test
    public void addPerformanceWithInvalidFeedStatus() {
        var feed = new Feed()
                .withClientId(client.getClientId().asLong())
                .withName("FEED_NAME")
                .withBusinessType(BusinessType.AUTO)
                .withFilename("feed.xml")
                .withOffersCount(20L)
                .withRefreshInterval(10L)
                .withSource(Source.FILE)
                .withUpdateStatus(UpdateStatus.ERROR);
        FeedInfo feedInfo = new FeedInfo().withClientInfo(client).withFeed(feed);
        Long feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var smartAdGroup = new SmartAdGroupAdd()
                .withFeedId(feedId)
                .withAdTitleSource(FEED_TITLE)
                .withAdBodySource(FEED_BODY);
        var adGroupAddItemMap = new AdGroupAddItemMap()
                .defaultAdGroupAddItem(performanceCampaignId)
                .withPerformanceAdGroup(smartAdGroup);

        api.userSteps().adGroupsSteps().shouldGetResultOnAdd(
                new AddRequestMap().withAdGroups(adGroupAddItemMap),
                ExpectedResult.errors(new Notification(5004, Api5ErrorDetails.FEED_STATUS_WRONG, UpdateStatus.ERROR))
        );
    }

    @Test
    @Ignore
    public void checkStatusAfterAdd() {
        Long id = api.userSteps().adGroupsSteps().addDefaultGroup(campaignId);
        GroupFakeInfo groupFakeInfo = api.userSteps().groupFakeSteps().getGroupParams(id);
        TestSteps.assertThat("StatusModerate = New", groupFakeInfo.getStatusModerate(), equalTo(Status.NEW));
    }
}
