package ru.yandex.autotests.direct.api.campaigns.add.dynamictextcampaign;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsArrayMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.PriorityGoalsItemMap;
import ru.yandex.autotests.directapi.model.campaigns.MetrikaGoals;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;
import ru.yandex.qatools.allure.annotations.Issue;

import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.nullValue;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;

@Aqua.Test
@Features(CampaignsFeatures.ADD)
@Description("Проверка создания кампании с PriorityGoals")
@Issue("https://st.yandex-team.ru/DIRECT-98633")
public class PriorityGoalsDBValuesTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();
    private static ClientInfo client;

    @BeforeClass
    public static void prepareData() {
        client = api.createClient();
        api.as(client);
        api.userSteps.featureSteps().setClientFeature(client, FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED, true);
    }

    @Test
    public void addWithoutPriorityGoals() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withDynamicTextCampaign(new DynamicTextCampaignAddItemMap()
                        .withBiddingStrategy(new DynamicTextCampaignStrategyAddMap()
                                .defaultDynamicTextCampaignStrategy()))
        );

        var dbMeaningfulGoals = getDbMeaningfulGoals(cid);
        assertThat("meaningful_goals пустой", dbMeaningfulGoals, nullValue());
    }

    @Test
    public void addWithEngagedSessionsPriorityGoal() {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withDynamicTextCampaign(new DynamicTextCampaignAddItemMap()
                        .withBiddingStrategy(new DynamicTextCampaignStrategyAddMap()
                                .defaultDynamicTextCampaignStrategy())
                        .withPriorityGoals(new PriorityGoalsArrayMap()
                                .withItems(new PriorityGoalsItemMap()
                                        .withEngagedSessionGoalId()
                                        .withValue(1_000_000L))))
        );

        var dbMeaningfulGoals = getDbMeaningfulGoals(cid);
        assertThat("meaningful_goals содержит правильный json",
                dbMeaningfulGoals,
                anyOf(
                        equalTo("[{\"goal_id\": " + PriorityGoalsItemMap.ENGAGED_SESSION_GOAL_ID + ", \"value\": " +
                                "\"1\"}]"),
                        equalTo("[{\"value\": \"1\", \"goal_id\": " + PriorityGoalsItemMap.ENGAGED_SESSION_GOAL_ID +
                                "}]"),
                        // через интерфейс фронт автоматом форматирует до двух знаков после запятой
                        // поэтому это норм, что и api5 будет делать так же (менять 1, на 1.00)
                        equalTo("[{\"goal_id\": " + PriorityGoalsItemMap.ENGAGED_SESSION_GOAL_ID + ", \"value\": \"1" +
                                ".00\"}]"),
                        equalTo("[{\"value\": \"1.00\", \"goal_id\": " + PriorityGoalsItemMap.ENGAGED_SESSION_GOAL_ID + "}]")
                ));
    }

    @Test
    public void addWithCounterGoal() {
        addAndCheckCampaignWithCounterGoal(MetrikaGoals.VALID_COUNTER_ID, MetrikaGoals.VALID_GOAL_ID);
    }

    @Test
    public void addWithInaccessibleCounterGoal() {
        addAndCheckCampaignWithCounterGoal(MetrikaGoals.INACCESSIBLE_COUNTER_ID, MetrikaGoals.INACCESSIBLE_GOAL_ID);
    }

    private void addAndCheckCampaignWithCounterGoal(int counterId, long goalId) {
        Long cid = api.userSteps.campaignSteps().addCampaign(new CampaignAddItemMap()
                .defaultCampaignAddItem()
                .withDynamicTextCampaign(new DynamicTextCampaignAddItemMap()
                        .withBiddingStrategy(new DynamicTextCampaignStrategyAddMap()
                                .defaultDynamicTextCampaignStrategy())
                        .withCounterIds(counterId)
                        .withPriorityGoals(new PriorityGoalsArrayMap()
                                .withItems(new PriorityGoalsItemMap()
                                        .withGoalId(goalId)
                                        .withValue(1_000_000L))))
        );

        var dbMeaningfulGoals = getDbMeaningfulGoals(cid);
        assertThat("meaningful_goals содержит правильный json",
                dbMeaningfulGoals,
                anyOf(
                        equalTo("[{\"goal_id\": " + goalId + ", \"value\": \"1\"}]"),
                        equalTo("[{\"value\": \"1\", \"goal_id\": " + goalId + "}]"),
                        // через интерфейс фронт автоматом форматирует до двух знаков после запятой
                        // поэтому это норм, что и api5 будет делать так же (менять 1, на 1.00)
                        equalTo("[{\"goal_id\": " + goalId + ", \"value\": \"1.00\"}]"),
                        equalTo("[{\"value\": \"1.00\", \"goal_id\": " + goalId + "}]")
                ));
    }

    private String getDbMeaningfulGoals(Long cid) {
        return api.userSteps.getDirectJooqDbSteps().useShardForClient(client).getContext()
                .select(CAMP_OPTIONS.MEANINGFUL_GOALS)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.eq(cid))
                .fetchSingle().value1();
    }
}
