package ru.yandex.autotests.direct.api.campaigns.add.dynamictextcampaign.strategy.payforconversioncrr;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.yandex.direct.api.v5.campaigns.CampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.CampaignGetItem;
import com.yandex.direct.api.v5.campaigns.CampaignsSelectionCriteria;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignFieldEnum;
import com.yandex.direct.api.v5.campaigns.DynamicTextCampaignSearchStrategyTypeEnum;
import org.jooq.Record2;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.aqua.annotations.project.Aqua;
import ru.yandex.autotests.direct.api.campaigns.CampaignsFeatures;
import ru.yandex.autotests.directapi.model.api5.campaigns.CampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignAddItemMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignNetworkStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignNetworkStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignSearchStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignSearchStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignStrategyAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.DynamicTextCampaignStrategyMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.GetRequestMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyPayForConversionCrrAddMap;
import ru.yandex.autotests.directapi.model.api5.campaigns.StrategyPayForConversionCrrMap;
import ru.yandex.autotests.directapi.model.campaigns.MetrikaGoals;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.test.utils.differ.BigDecimalDiffer;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static org.hamcrest.Matchers.anything;
import static org.hamcrest.Matchers.equalTo;
import static ru.yandex.autotests.directapi.matchers.beandiffer2.BeanDifferMatcherV5.beanDifferV5;
import static ru.yandex.autotests.directapi.model.api5.campaigns.StrategyPayForConversionCrrAddMap.DEFAULT_CRR;
import static ru.yandex.autotests.directapi.model.api5.campaigns.StrategyPayForConversionCrrAddMap.DEFAULT_WEEKLY_SPEND_LIMIT;
import static ru.yandex.autotests.directapi.model.api5.campaigns.StrategyPayForConversionCrrAddMap.MAX_CRR;
import static ru.yandex.autotests.directapi.model.api5.campaigns.StrategyPayForConversionCrrAddMap.MIN_CRR;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;
import static ru.yandex.autotests.irt.testutils.beandiffer2.beanfield.BeanFieldPath.newPath;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;

@Aqua.Test
@Features(CampaignsFeatures.ADD)
@Description("Позитивные сценарии проверки создания до со стратегией PayForConversionCrr")
@RunWith(Parameterized.class)
public class StrategyPayForConversionCrrPositiveTest {
    @ClassRule
    public static final ApiSteps api = new ApiSteps();
    private static final Currency currency = CurrencyCode.RUB.getCurrency();
    private static final Money minLimit = Money.valueOf(
            currency.getMinAutobudget(),
            currency.getCode());
    private static final Money maxLimit = Money.valueOf(
            currency.getMaxAutobudget(),
            currency.getCode());
    private static ClientInfo client;


    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public Long goalId;

    @Parameterized.Parameter(value = 2)
    public Integer crr;

    @Parameterized.Parameter(value = 3)
    public Long weeklySpendLimit;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection<?> testData() {
        Object[][] data = new Object[][]{
                {
                        "Создание новой кампании",
                        MetrikaGoals.VALID_GOAL_ID,
                        DEFAULT_CRR,
                        DEFAULT_WEEKLY_SPEND_LIMIT,
                },
                {
                        "Создание новой кампании с минимальным ДРР",
                        MetrikaGoals.VALID_GOAL_ID,
                        MIN_CRR,
                        DEFAULT_WEEKLY_SPEND_LIMIT,
                },
                {
                        "Создание новой кампании с максимальным ДРР",
                        MetrikaGoals.VALID_GOAL_ID,
                        MAX_CRR,
                        DEFAULT_WEEKLY_SPEND_LIMIT,
                },
                {
                        "Создание новой кампании с минимальным бюджетом",
                        MetrikaGoals.VALID_GOAL_ID,
                        DEFAULT_CRR,
                        minLimit.micros(),
                },
                {
                        "Создание новой кампании с максимальным бюджетом",
                        MetrikaGoals.VALID_GOAL_ID,
                        DEFAULT_CRR,
                        maxLimit.micros(),
                },
                {
                        "Создание новой кампании без недельного бюджета",
                        MetrikaGoals.VALID_GOAL_ID,
                        DEFAULT_CRR,
                        null
                }
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void prepareData() {
        client = api.createClient();
        api.as(client);

        api.userSteps.featureSteps().ensureClientHasFeature(client, FeatureName.CRR_STRATEGY_ALLOWED);
        api.userSteps.featureSteps().ensureClientHasFeature(client, FeatureName.FIX_CRR_STRATEGY_ALLOWED);
        api.userSteps.featureSteps().ensureClientHasFeature(client, FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED);
    }

    @Test
    public void shouldCreateFixCrrCampaign() {
        StrategyData expectedStrategyData = new StrategyData()
                .withName(CampaignsStrategyName.autobudget_crr.name())
                .withGoalId(goalId)
                .withCrr(crr.longValue())
                .withPayForConversion(true)
                .withVersion(1L);
        if (weeklySpendLimit != null) {
            expectedStrategyData.setSum(Money.valueOfMicros(weeklySpendLimit, currency.getCode()).bigDecimalValue());
        }

        DynamicTextCampaignStrategyMap expectedStrategy = new DynamicTextCampaignStrategyMap()
                .withSearch(new DynamicTextCampaignSearchStrategyMap()
                        .withBiddingStrategyType(DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR)
                        .withPayForConversionCrr(new StrategyPayForConversionCrrMap()
                                .withGoalId(goalId)
                                .withCrr(crr)
                                .withWeeklySpendLimit(weeklySpendLimit)
                        )
                ).withNetwork(new DynamicTextCampaignNetworkStrategyMap().defaultServingOff());

        long cid = api.userSteps.campaignSteps()
                .addCampaign(new CampaignAddItemMap()
                        .defaultCampaignAddItem()
                        .withDynamicTextCampaign(new DynamicTextCampaignAddItemMap()
                                .withBiddingStrategy(new DynamicTextCampaignStrategyAddMap()
                                        .withSearch(new DynamicTextCampaignSearchStrategyAddMap()
                                                .withBiddingStrategyType(DynamicTextCampaignSearchStrategyTypeEnum.PAY_FOR_CONVERSION_CRR)
                                                .withPayForConversionCrr(new StrategyPayForConversionCrrAddMap()
                                                        .withWeeklySpendLimit(weeklySpendLimit)
                                                        .withCrr(crr)
                                                        .withGoalId(goalId)
                                                )
                                        ).withNetwork(new DynamicTextCampaignNetworkStrategyAddMap().defaultServingOff())
                                )
                        )
                );


        Record2<String, CampaignsStrategyName> campaignRecord =
                api.userSteps.getDirectJooqDbSteps().useShardForClient(client)
                        .getContext()
                        .select(CAMPAIGNS.STRATEGY_DATA, CAMPAIGNS.STRATEGY_NAME)
                        .from(CAMPAIGNS)
                        .where(CAMPAIGNS.CID.eq(cid))
                        .fetchAny();

        assertThat("Кампания в БД со стратегией AUTOBUDGET_CRR и payForConversion = 1",
                JsonUtils.fromJson(campaignRecord.value1(), StrategyData.class),
                beanDiffer(expectedStrategyData)
                        .useCompareStrategy(
                                DefaultCompareStrategies.onlyExpectedFields()
                                        .forFields(newPath("strategy_data/last_bidder_restart_time"))
                                        .useMatcher(anything())
                                        .forFields(newPath("sum"))
                                        .useDiffer(new BigDecimalDiffer()))
        );
        assertThat("Кампания в БД со стратегией AUTOBUDGET_CRR",
                campaignRecord.value2(),
                equalTo(CampaignsStrategyName.autobudget_crr)
        );

        List<CampaignGetItem> campaignsFromApi = api.userSteps.campaignSteps().getCampaigns(
                new GetRequestMap().withSelectionCriteria(new CampaignsSelectionCriteria()
                                .withIds(cid)
                        )
                        .withFieldNames(CampaignFieldEnum.ID)
                        .withDynamicTextCampaignFieldNames(DynamicTextCampaignFieldEnum.BIDDING_STRATEGY)
        );

        assertThat("Получили корректную кампанию через API",
                campaignsFromApi.get(0).getDynamicTextCampaign().getBiddingStrategy(),
                beanDifferV5(expectedStrategy.getBean())
        );
    }
}
