package ru.yandex.autotests.direct.api.feeds.add;

import com.yandex.direct.api.v5.feeds.AddRequest;
import com.yandex.direct.api.v5.feeds.BusinessTypeEnum;
import com.yandex.direct.api.v5.feeds.FeedAddItem;
import com.yandex.direct.api.v5.feeds.FileFeedAdd;
import com.yandex.direct.api.v5.feeds.SourceTypeEnum;
import com.yandex.direct.api.v5.feeds.UrlFeedAdd;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.api.feeds.FeedsFeatures;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.qatools.allure.annotations.Features;

@Features(FeedsFeatures.ADD)
public class AddFeedsTest {

    private static final String FEED_URL = "http://ya.ru/feed.csv";
    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);

        api.userSteps().clientFakeSteps().fakeClearClientSpentUnits(clientInfo);
    }

    @Test
    public void add_file_success() {
        FeedAddItem item = getFileFeedAddItem("filename.csv");
        var addRequest = new AddRequest().withFeeds(item);

        api.userSteps().feedSteps().shouldGetResultOnAdd(
                addRequest,
                ExpectedResult.success()
        );
    }

    @Test
    public void add_url_success() {
        FeedAddItem item = getUrlFeedAddItem();
        var addRequest = new AddRequest().withFeeds(item);

        api.userSteps().feedSteps().shouldGetResultOnAdd(
                addRequest,
                ExpectedResult.success()
        );
    }

    @Test
    public void add_invalidFilename_failure() {
        FeedAddItem item = getFileFeedAddItem("filename.jpg");
        var addRequest = new AddRequest().withFeeds(item);

        api.userSteps().feedSteps().shouldGetResultOnAdd(
                addRequest,
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FEED_INVALID_FILENAME))
        );
    }

    @Test
    public void add_invalidUrl_failure() {
        var urlFeed = new UrlFeedAdd().withUrl("invalid.url");
        FeedAddItem item = getUrlFeedAddItem(urlFeed);
        var addRequest = new AddRequest().withFeeds(item);

        api.userSteps().feedSteps().shouldGetResultOnAdd(
                addRequest,
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FEED_INVALID_HREF))
        );
    }

    @Test
    public void add_emptyName_failure() {
        FeedAddItem item = getUrlFeedAddItem().withName("");
        var addRequest = new AddRequest().withFeeds(item);

        api.userSteps().feedSteps().shouldGetResultOnAdd(
                addRequest,
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FEED_EMPTY_NAME))
        );
    }

    @Test
    public void add_loginWithoutPassword_failure() {
        var urlFeed = new UrlFeedAdd().withUrl("http://url.ru").withLogin("login");
        FeedAddItem item = getUrlFeedAddItem(urlFeed);
        var addRequest = new AddRequest().withFeeds(item);

        api.userSteps().feedSteps().shouldGetResultOnAdd(
                addRequest,
                ExpectedResult.errors(new Notification(5000, Api5ErrorDetails.FEED_PASSWORD_IS_NOT_SET))
        );
    }

    @Test
    public void add_passwordWithoutLogin_failure() {
        var urlFeed = new UrlFeedAdd().withUrl("http://url.ru").withPassword("password");
        FeedAddItem item = getUrlFeedAddItem(urlFeed);
        var addRequest = new AddRequest().withFeeds(item);

        api.userSteps().feedSteps().shouldGetResultOnAdd(
                addRequest,
                ExpectedResult.errors(new Notification(5000, Api5ErrorDetails.FEED_LOGIN_IS_NOT_SET))
        );
    }

    private FeedAddItem getFileFeedAddItem(String filename) {
        var fileFeed = new FileFeedAdd()
                .withFilename(filename)
                .withData("<xml></xml>".getBytes());
        return new FeedAddItem()
                .withName("name")
                .withBusinessType(BusinessTypeEnum.FLIGHTS)
                .withSourceType(SourceTypeEnum.FILE)
                .withFileFeed(fileFeed);
    }

    private FeedAddItem getUrlFeedAddItem() {
        var urlFeed = new UrlFeedAdd().withUrl(FEED_URL);
        return getUrlFeedAddItem(urlFeed);
    }

    private FeedAddItem getUrlFeedAddItem(UrlFeedAdd urlFeed) {
        return new FeedAddItem()
                .withName("name")
                .withBusinessType(BusinessTypeEnum.REALTY)
                .withSourceType(SourceTypeEnum.URL)
                .withUrlFeed(urlFeed);
    }
}
