package ru.yandex.autotests.direct.api.feeds.delete;

import com.yandex.direct.api.v5.feeds.DeleteRequest;
import com.yandex.direct.api.v5.feeds.FeedsSelectionCriteria;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.api.feeds.FeedsFeatures;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.info.FeedInfo;
import ru.yandex.qatools.allure.annotations.Features;

import static ru.yandex.autotests.direct.api.feeds.TestFeeds.buildFileFeed;

@Features(FeedsFeatures.DELETE)
public class DeleteFeedsTest {
    private static final Long NON_EXISTENT_FEED_ID = 2342L;

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    private static ClientInfo clientInfo;

    @BeforeClass
    public static void init() {
        clientInfo = api.createClient();
        api.as(clientInfo);

        api.userSteps().clientFakeSteps().fakeClearClientSpentUnits(clientInfo);
    }

    @Test
    public void delete_notUsed_success() {
        var feed = buildFileFeed(clientInfo.getClientId()).withUpdateStatus(UpdateStatus.ERROR);
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var selectionCriteria = new FeedsSelectionCriteria().withIds(feedId);
        var deleteRequest = new DeleteRequest().withSelectionCriteria(selectionCriteria);

        api.userSteps().feedSteps().shouldGetResultOnDelete(
                deleteRequest,
                ExpectedResult.success()
        );
    }

    @Test
    public void delete_used_failure() {
        var feed = buildFileFeed(clientInfo.getClientId()).withUpdateStatus(UpdateStatus.DONE);
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        Long campaignId = api.userSteps().campaignSteps().addDefaultSmartCampaign(clientInfo);
        api.userSteps().adGroupsSteps().addDefaultPerformanceGroup(campaignId, feedId);
        var selectionCriteria = new FeedsSelectionCriteria().withIds(feedId);
        var deleteRequest = new DeleteRequest().withSelectionCriteria(selectionCriteria);

        api.userSteps().feedSteps().shouldGetResultOnDelete(
                deleteRequest,
                ExpectedResult.errors(new Notification(5004, Api5ErrorDetails.FEED_USED_IN_AD_GROUP))
        );
    }

    @Test
    public void delete_notUsedButPrepareToBannerLand_failure() {
        var feed = buildFileFeed(clientInfo.getClientId()).withUpdateStatus(UpdateStatus.NEW);
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var selectionCriteria = new FeedsSelectionCriteria().withIds(feedId);
        var deleteRequest = new DeleteRequest().withSelectionCriteria(selectionCriteria);

        api.userSteps().feedSteps().shouldGetResultOnDelete(
                deleteRequest,
                ExpectedResult.errors(new Notification(5004, Api5ErrorDetails.FEED_STATUS_WRONG, UpdateStatus.NEW))
        );
    }

    @Test
    public void delete_nonExistent_failure() {
        var selectionCriteria = new FeedsSelectionCriteria().withIds(NON_EXISTENT_FEED_ID);
        var deleteRequest = new DeleteRequest()
                .withSelectionCriteria(selectionCriteria);

        api.userSteps().feedSteps().shouldGetResultOnDelete(
                deleteRequest,
                ExpectedResult.errors(new Notification(8800, Api5ErrorDetails.FEED_NOT_EXIST_AD_GROUP,
                        NON_EXISTENT_FEED_ID))
        );
    }
}
