package ru.yandex.autotests.direct.api.feeds.get;

import com.yandex.direct.api.v5.feeds.BusinessTypeEnum;
import com.yandex.direct.api.v5.feeds.FeedFieldEnum;
import com.yandex.direct.api.v5.feeds.FeedGetItem;
import com.yandex.direct.api.v5.feeds.FeedStatusEnum;
import com.yandex.direct.api.v5.feeds.FeedsSelectionCriteria;
import com.yandex.direct.api.v5.feeds.GetRequest;
import com.yandex.direct.api.v5.feeds.GetResponse;
import org.hamcrest.CoreMatchers;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.api.feeds.FeedsFeatures;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.info.FeedInfo;
import ru.yandex.qatools.allure.annotations.Features;

import static ru.yandex.autotests.direct.api.feeds.TestFeeds.buildFileFeed;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Features(FeedsFeatures.GET)
public class GetFeedsTest {
    private static final Long NON_EXISTENT_FEED_ID = 2342L;
    private static final String FEED_NAME = "feed name";

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    private static ClientInfo clientInfo;

    @BeforeClass
    public static void init() {
        clientInfo = api.createClient();
        api.as(clientInfo);

        api.userSteps().clientFakeSteps().fakeClearClientSpentUnits(clientInfo);
        api.userSteps().campaignSteps().addDefaultSmartCampaign(clientInfo);
    }

    @Test
    public void get_file_success() {
        var feed = buildFileFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var selectionCriteria = new FeedsSelectionCriteria().withIds(feedId);
        var getRequest = new GetRequest()
                .withSelectionCriteria(selectionCriteria)
                .withFieldNames(FeedFieldEnum.ID, FeedFieldEnum.NAME, FeedFieldEnum.STATUS,
                        FeedFieldEnum.BUSINESS_TYPE);

        var actual = api.userSteps().feedSteps().feedsGet(getRequest);
        var feedGetItem = new FeedGetItem()
                .withId(feedId)
                .withName(FEED_NAME)
                .withStatus(FeedStatusEnum.DONE)
                .withBusinessType(BusinessTypeEnum.AUTOMOBILES);
        var expected = new GetResponse().withFeeds(feedGetItem);
        assertThat("Полученный фид совпадает с ожидаемым", actual, beanDiffer(expected));
    }

    @Test
    public void get_all_success() {
        var feed = buildFileFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var getRequest = new GetRequest()
                .withFieldNames(FeedFieldEnum.ID);

        GetResponse response = api.userSteps().feedSteps().feedsGet(getRequest);
        var returnedIds = mapList(response.getFeeds(), FeedGetItem::getId);
        assertThat("Содержит фид", returnedIds, CoreMatchers.hasItem(feedId));
    }

    @Test
    public void get_nonExistent() {
        var selectionCriteria = new FeedsSelectionCriteria().withIds(NON_EXISTENT_FEED_ID);
        var getRequest = new GetRequest()
                .withSelectionCriteria(selectionCriteria)
                .withFieldNames(FeedFieldEnum.ID);

        var actual = api.userSteps().feedSteps().feedsGet(getRequest);
        var expected = new GetResponse();
        assertThat("Получен пустой ответ", actual, beanDiffer(expected));
    }
}
