package ru.yandex.autotests.direct.api.feeds.update;

import com.yandex.direct.api.v5.feeds.FeedUpdateItem;
import com.yandex.direct.api.v5.feeds.FileFeedUpdate;
import com.yandex.direct.api.v5.feeds.ObjectFactory;
import com.yandex.direct.api.v5.feeds.UpdateRequest;
import com.yandex.direct.api.v5.feeds.UrlFeedUpdate;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.api.feeds.FeedsFeatures;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.info.FeedInfo;
import ru.yandex.qatools.allure.annotations.Features;

import static ru.yandex.autotests.direct.api.feeds.TestFeeds.buildFileFeed;
import static ru.yandex.autotests.direct.api.feeds.TestFeeds.buildUrlFeed;

@Features(FeedsFeatures.UPDATE)
public class UpdateFeedsTest {

    private static final ObjectFactory OBJECT_FACTORY = new ObjectFactory();
    private static final String FEED_URL = "http://newurl.ru/feed.zip";

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    private static ClientInfo clientInfo;

    @BeforeClass
    public static void init() {
        clientInfo = api.createClient();
        api.as(clientInfo);

        api.userSteps().clientFakeSteps().fakeClearClientSpentUnits(clientInfo);
    }

    @Test
    public void update_file_success() {
        var feed = buildFileFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var fileFeed = new FileFeedUpdate()
                .withFilename("newFilename.csv")
                .withData("<xml></xml>".getBytes());
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withName("newName")
                .withFileFeed(fileFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.success()
        );
    }

    @Test
    public void update_url_success() {
        var feed = buildUrlFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withUrl(FEED_URL)
                .withLogin(OBJECT_FACTORY.createUrlFeedUpdateLogin("newLogin"))
                .withPassword(OBJECT_FACTORY.createUrlFeedUpdatePassword("newPassword"));
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withName("newName")
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.success()
        );
    }

    @Test
    public void update_fileWithInvalidValues_failure() {
        var feed = buildFileFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var fileFeed = new FileFeedUpdate()
                .withFilename("newFilename.jpg")
                .withData("<xml></xml>".getBytes());
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withName("newName")
                .withFileFeed(fileFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FEED_INVALID_FILENAME))
        );
    }

    @Test
    public void update_urlWithInvalidValues_failure() {
        var feed = buildUrlFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withUrl("invalid.url");
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withName("")
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.FEED_EMPTY_NAME),
                        new Notification(5005, Api5ErrorDetails.FEED_INVALID_HREF)
                )
        );
    }

    @Test
    public void update_loginWithoutPassword_failure() {
        var feed = buildUrlFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withLogin(OBJECT_FACTORY.createUrlFeedUpdateLogin("login"));
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FEED_APART_RESET_LOGIN_AND_PASSWORD))
        );
    }

    @Test
    public void update_passwordWithoutLogin_failure() {
        var feed = buildUrlFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withPassword(OBJECT_FACTORY.createUrlFeedUpdatePassword("password"));
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FEED_APART_RESET_LOGIN_AND_PASSWORD))
        );
    }

    @Test
    public void update_loginWithoutPasswordButExists_success() {
        var feed = buildUrlFeed(clientInfo.getClientId())
                .withLogin("login")
                .withPlainPassword("password");
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withLogin(OBJECT_FACTORY.createUrlFeedUpdateLogin("login2"));
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.success()
        );
    }

    @Test
    public void update_passwordWithoutLoginButExists_success() {
        var feed = buildUrlFeed(clientInfo.getClientId())
                .withLogin("login")
                .withPlainPassword("password");
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withPassword(OBJECT_FACTORY.createUrlFeedUpdatePassword("password2"));
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.success()
        );
    }

    @Test
    public void update_urlAsFile_failure() {
        var feed = buildUrlFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        var fileFeed = new FileFeedUpdate()
                .withFilename("newFilename.jpg")
                .withData("<xml></xml>".getBytes());
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withFileFeed(fileFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.errors(new Notification(6000, Api5ErrorDetails.FEED_INCONSISTENT_TYPE))
        );
    }

    @Test
    public void update_fileAsUrl_failure() {
        var feed = buildFileFeed(clientInfo.getClientId());
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withUrl(FEED_URL);
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.errors(new Notification(6000, Api5ErrorDetails.FEED_INCONSISTENT_TYPE))
        );
    }

    @Test
    public void update_resetPasswordWithoutLogin_failure() {
        var feed = buildUrlFeed(clientInfo.getClientId())
                .withLogin("login")
                .withPlainPassword("password");
        var feedInfo = new FeedInfo().withFeed(feed).withClientInfo(clientInfo);
        var feedId = api.userSteps().feedSteps().createFeed(feedInfo).getFeedId();
        UrlFeedUpdate urlFeed = new UrlFeedUpdate()
                .withPassword(OBJECT_FACTORY.createUrlFeedUpdatePassword(null));
        FeedUpdateItem item = new FeedUpdateItem()
                .withId(feedId)
                .withUrlFeed(urlFeed);
        var updateRequest = new UpdateRequest().withFeeds(item);
        api.userSteps().feedSteps().shouldGetResultOnUpdate(
                updateRequest,
                ExpectedResult.errors(new Notification(5005, Api5ErrorDetails.FEED_APART_RESET_LOGIN_AND_PASSWORD))
        );
    }
}
