package ru.yandex.autotests.direct.api.keywordsresearch.deduplicate;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import com.google.common.collect.ImmutableList;
import com.yandex.direct.api.v5.general.IdsCriteria;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateOperationEnum;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponse;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseAddItem;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseUpdateItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.autotests.direct.api.keywordsresearch.KeywordsResearchFeatures;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestItemMap;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestMap;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Features(KeywordsResearchFeatures.DEDUPLICATE)
@Description("Тест на общую корректность работы сервиса в режиме работы по умолчанию")
@RunWith(Parameterized.class)
public class DeduplicateDefaultModeTest {

    @ClassRule
    public static final ApiSteps api = new ApiSteps();

    @Parameterized.Parameter(0)
    public List<DeduplicateOperationEnum> operation;

    @Parameterized.Parameters(name = "deduplicateMode = {0}")
    public static Collection fieldsValue() {
        Object[][] data = new Object[][] {
                { null },
                { emptyList() },
                { ImmutableList.of(DeduplicateOperationEnum.MERGE_DUPLICATES,
                        DeduplicateOperationEnum.ELIMINATE_OVERLAPPING) }
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);
    }

    @Test
    public void convertToAddTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(new DeduplicateRequestItemMap().withKeyword("ключевая фраза"));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseAddItem keywordAdd = new DeduplicateResponseAddItem();
        keywordAdd.setKeyword("ключевая фраза");

        expected.setAdd(singletonList(keywordAdd));

        assertThat("Добавляется корректный текст", actual, beanDiffer(expected));
    }

    @Test
    public void convertToUpdateTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(1L),
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза -тест"));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseUpdateItem keywordUpdate = new DeduplicateResponseUpdateItem();
        keywordUpdate.setKeyword("ключевая фраза -тест");
        keywordUpdate.setId(1L);

        expected.setUpdate(singletonList(keywordUpdate));

        assertThat("Обновляется текст на корректный", actual, beanDiffer(expected));
    }

    @Test
    public void convertToDeleteTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза -тест").withId(1L),
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(2L));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        IdsCriteria ids = new IdsCriteria();
        ids.setIds(singletonList(2L));

        expected.setDelete(ids);

        assertThat("Удаляется корректная фраза", actual, beanDiffer(expected));
    }

    @Test
    public void convertToFailureTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("ключевая++ фраза"));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        assumeThat("ответ Failure из сервиса получен", actual.getFailure(),
                hasSize(1));
        assertThat("Возвращается корректная позиция ошибки", actual.getFailure().get(0).getPosition(),
                equalTo(1L));
    }

    @Test
    public void unglueTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("ключевая").withId(1L),
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(2L));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseUpdateItem keywordUpdate = new DeduplicateResponseUpdateItem();
        keywordUpdate.setKeyword("ключевая -фраза");
        keywordUpdate.setId(1L);

        expected.setUpdate(singletonList(keywordUpdate));

        assertThat("Ключевые фразы расклеены", actual, beanDiffer(expected));
    }

    @Test
    public void glueSameKeywords() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза -раз"),
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза -два").withId(2L));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseUpdateItem keywordUpdate = new DeduplicateResponseUpdateItem();
        keywordUpdate.setKeyword("ключевая фраза -два -раз");
        keywordUpdate.setId(2L);

        expected.setUpdate(singletonList(keywordUpdate));

        assertThat("Ключевые фразы склеены", actual, beanDiffer(expected));
    }

    @Test
    public void glueDifferentKeywordsWithSameNormalizedValue() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("окна на заказ -дешево"),
                    new DeduplicateRequestItemMap().withKeyword("заказ окон -жалюзи"));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseAddItem keywordAdd = new DeduplicateResponseAddItem();
        keywordAdd.setKeyword("заказ окон -дешево -жалюзи");

        expected.setAdd(singletonList(keywordAdd));

        assertThat("Ключевые фразы склеены", actual, beanDiffer(expected));
    }

    @Test
    public void minusWordsReturnedInOriginalState() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(new DeduplicateRequestItemMap().withKeyword("куплю -слона"));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseAddItem keyword = new DeduplicateResponseAddItem();
        keyword.setKeyword("куплю -слона");

        expected.setAdd(singletonList(keyword));

        assertThat("Минус-слова возвращаются в исходной форме", actual, beanDiffer(expected));
    }

    @Test
    public void weightTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(1L).withWeight(100L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -два").withId(2L).withWeight(1L));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseUpdateItem update = new DeduplicateResponseUpdateItem();
        update.setKeyword("ключевая фраза -два");
        update.setId(1L);

        IdsCriteria ids = new IdsCriteria();
        ids.setIds(singletonList(2L));

        expected.setUpdate(singletonList(update));
        expected.setDelete(ids);

        assertThat("Веса обрабатываются корректно", actual, beanDiffer(expected));
    }

    @Test
    public void anotherWeightsWeightTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(1L).withWeight(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -два").withId(2L).withWeight(100L));

        if (operation != null) {
            request.withOperations(operation);
        }

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        IdsCriteria ids = new IdsCriteria();
        ids.setIds(singletonList(1L));

        expected.setDelete(ids);

        assertThat("Веса обрабатываются корректно", actual, beanDiffer(expected));
    }
}
