package ru.yandex.autotests.direct.api.keywordsresearch.deduplicate;

import com.yandex.direct.api.v5.general.IdsCriteria;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateOperationEnum;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponse;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseAddItem;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseUpdateItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.api.keywordsresearch.KeywordsResearchFeatures;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestItemMap;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestMap;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Features(KeywordsResearchFeatures.DEDUPLICATE)
@Description("Тест на общую корректность работы сервиса с операцией ELIMINATE_OVERLAPPING")
public class DeduplicateEliminateOverlappingModeTest {

    @ClassRule
    public static final ApiSteps api = new ApiSteps();

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);
    }

    @Test
    public void convertToAddTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(new DeduplicateRequestItemMap().withKeyword("ключевая фраза"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem();
        keyword.setKeyword("ключевая фраза");

        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Добавляется корректный текст", actual, beanDiffer(expected));
    }

    @Test
    public void convertToUpdateTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keywordUpdate = new DeduplicateResponseUpdateItem();
        keywordUpdate.setKeyword("ключевая -фраза");
        keywordUpdate.setId(1L);

        var keywordAdd = new DeduplicateResponseAddItem();
        keywordAdd.setKeyword("ключевая фраза");

        var expected = new DeduplicateResponse();
        expected.setUpdate(singletonList(keywordUpdate));
        expected.setAdd(singletonList(keywordAdd));

        assertThat("Обновляется текст на корректный", actual, beanDiffer(expected));
    }

    @Test
    public void convertToDeleteTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(1L),
                    new DeduplicateRequestItemMap().withKeyword("ключевая").withId(2L),
                    new DeduplicateRequestItemMap().withKeyword("ключевая -фраза").withId(3L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        IdsCriteria ids = new IdsCriteria();
        ids.setIds(singletonList(3L));

        var keywordUpdate = new DeduplicateResponseUpdateItem();
        keywordUpdate.setKeyword("ключевая -фраза");
        keywordUpdate.setId(2L);

        var expected = new DeduplicateResponse();
        expected.setUpdate(singletonList(keywordUpdate));
        expected.setDelete(ids);

        assertThat("Удаляется корректная фраза", actual, beanDiffer(expected));
    }

    @Test
    public void convertToFailureTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая++ фраза"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        assumeThat("Ответ Failure из сервиса получен", actual.getFailure(),
                hasSize(1));
        assertThat("Возвращается корректная позиция ошибки", actual.getFailure().get(0).getPosition(),
                equalTo(1L));
    }

    @Test
    public void unglueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("дача квартира -ремонт").withId(1L),
                    new DeduplicateRequestItemMap().withKeyword("квартира машина дача").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseUpdateItem();
        keyword.setKeyword("дача квартира -ремонт -машина");
        keyword.setId(1L);

        var expected = new DeduplicateResponse().withUpdate(singletonList(keyword));

        assertThat("Ключевые фразы расклеены", actual, beanDiffer(expected));
    }

    @Test
    public void unglueOverlappingLemmasTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("Септик для высоких грунтовых").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("Септик грунтовый").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseUpdateItem();
        keyword.setKeyword("Септик грунтовый -высоких");
        keyword.setId(2L);

        var expected = new DeduplicateResponse().withUpdate(singletonList(keyword));

        assertThat("Ключевые фразы со словами, имеющими пересекающиеся леммы, расклеены", actual,
                beanDiffer(expected));
    }

    @Test
    public void noGlueWhenSamePlusWordsTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("дача квартира -ремонт").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("дача квартира -гостиница").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var expected = new DeduplicateResponse();

        assertThat("Ключевые фразы не должны склеиваться", actual, beanDiffer(expected));
    }

    @Test
    public void noGlueWhenDifferentPlusWordsWithSameNormalizedValueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("окна на заказ -дешево").withId(1L),
                    new DeduplicateRequestItemMap().withKeyword("заказ окон -жалюзи").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var expected = new DeduplicateResponse();

        assertThat("Ключевые фразы не должны склеиваться", actual, beanDiffer(expected));
    }

    @Test
    public void minusWordsReturnInOriginalStateTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(new DeduplicateRequestItemMap().withKeyword("куплю -слона"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("куплю -слона");
        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Минус-слова возвращаются в исходной форме", actual, beanDiffer(expected));
    }

    @Test
    public void minusWordsReturnInOriginalStateWhenUngluedTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("куплю слона").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("куплю индийского слона").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseUpdateItem();
        keyword.setKeyword("куплю слона -индийского");
        keyword.setId(1L);

        var expected = new DeduplicateResponse().withUpdate(singletonList(keyword));

        assertThat("Минус-слова возвращаются в исходной форме после расклейки", actual,
                beanDiffer(expected));
    }

    @Test
    public void weightTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("ключевая").withId(1L).withWeight(3L),
                    new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(2L).withWeight(2L),
                    new DeduplicateRequestItemMap().withKeyword("ключевая -фраза").withId(3L).withWeight(1L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var update = new DeduplicateResponseUpdateItem();
        update.setKeyword("ключевая -фраза");
        update.setId(1L);

        var ids = new IdsCriteria().withIds(singletonList(3L));

        var expected = new DeduplicateResponse();
        expected.setUpdate(singletonList(update));
        expected.setDelete(ids);

        assertThat("Веса обрабатываются корректно", actual, beanDiffer(expected));
    }

    @Test
    public void anotherWeightsWeightTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.ELIMINATE_OVERLAPPING)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая").withId(1L).withWeight(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза").withId(2L).withWeight(2L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая -фраза").withId(3L).withWeight(3L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var ids = new IdsCriteria().withIds(singletonList(1L));
        var expected = new DeduplicateResponse().withDelete(ids);

        assertThat("Веса обрабатываются корректно", actual, beanDiffer(expected));
    }
}
