package ru.yandex.autotests.direct.api.keywordsresearch.deduplicate;

import java.util.Arrays;
import java.util.Collection;

import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.autotests.direct.api.keywordsresearch.KeywordsResearchFeatures;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestItemMap;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestMap;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static org.apache.commons.lang3.StringUtils.capitalize;

@Features(KeywordsResearchFeatures.DEDUPLICATE)
@Description("Негативные запросы с минус-словами")
@RunWith(Parameterized.class)
public class DeduplicateKeywordValidationNegativeTest {

    @ClassRule
    public static final ApiSteps api = new ApiSteps();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String keyword;

    @Parameterized.Parameter(value = 2)
    public Notification notification;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection fields() {
        Object[][] data = new Object[][] {
                {
                        "Ключевые слова содержат минус-слово", "доставка пицца -пицца",
                        new Notification(5161,
                                Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS_IN_KEYWORDS, "пицца")
                },
                {
                        "Ключевые слова содержат минус-слово", "пицца -пицца",
                        new Notification(5161,
                                Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS_IN_KEYWORDS, "пицца")
                },
                {
                        "Ключевое слово предваряется минус-словом", "высокий -полет квадрокоптер",
                        new Notification(5161,
                                Api5ErrorDetailsJava.ONLY_SEPARATE_MINUS_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORDS)
                },
                {
                        "Минус-слово без ключевого", "-балкон",
                        new Notification(5161,
                                Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY)
                },
                {
                        "Простая группировка с минус-словами", "сервисное обслуживание (Kia | Ford) -гарантийное -бесплатное",
                        new Notification(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                capitalize(DeduplicateRequestItemMap.KEYWORD))
                },
                {
                        "2 точки в минус-слове", "пицца -57.21.45",
                        new Notification(5161,
                                Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORDS_PHRASES)
                },
                {
                        "Минус-слово в кавычках", "\"-пицца\"",
                        new Notification(5161, Api5ErrorDetailsJava.BOTH_QUOTES_AND_MINUS_WORDS_IN_KEYWORDS)
                },
                {
                        "Неправильное использование специальных символов", "[.пицца -доставка",
                        new Notification(5002, Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS,
                                "[.пицца -доставка")
                },
                {
                        "Неправильное использование специальных символов", "пицца -'доставка -гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Минус-слово с точкой", "гостиница -пиц.ца -доставка",
                        new Notification(5161,
                                Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORDS_PHRASES)
                },
                {
                        "Неправильное использование знака +", "пицца доставка++отель гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака +", "пицца доставка+ гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака +", "пицца доставка гостиница+",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака +", "пицца доставка+гостиница отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака -", "пицца отель--гостиница доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака -", "пицца доставка- гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака -", "пицца доставка отель-",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака !", "пицца отель !-доставка гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака !", "пицца отель X-!доставка гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака !", "пицца доставка -!-отель гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака +", "гостиница отель +-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака +", "гостиница отель X-+пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Неправильное использование знака +", "гостиница отель -+-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Восклицательный знак с другим модификатором", "гостиница !\"отель-пицца\" доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORDS)
                },
                {
                        "Восклицательный знак в конце фразы", "гостиница отель-пицца доставка!",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORDS)
                },
                {
                        "Восклицательный знак без слова", "гостиница ! отель-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORDS)
                },
                {
                        "Два восклицательных знака подряд", "гостиница !!отель-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORDS)
                },
                {
                        "Знак минус внутри квадратных скобок", "пицца [-отель] доставка",
                        new Notification(5002, Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS,
                                "пицца [-отель] доставка")
                },
                {
                        "Модификаторы внутри квадратных скобок", "пицца [+\"отель\"] доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORDS)
                },
                {
                        "Вложенные квадратные скобки", "пицца [ [отель] доставка]",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Пустые квадратные скобки", "пицца [] отель доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Квадратные скобки в неправильном порядке", "]пицца[",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [отель [доставка гостиница]",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца гостиница] отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [гостиница отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORDS)
                },
                {
                        "8 ключевых слов", "пицца отель москва гостиница поезд самолет такси теплоход",
                        new Notification(5140, Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONTAIN_MORE_WORDS,
                                PhraseConstraints.WORDS_MAX_COUNT)
                },
                {
                        "Слишком длинное ключевое слово", "отель пииииииииииииииииииииииииииииииииииииииииииицца теплоход",
                        new Notification(5142, Api5ErrorDetailsJava.KEYWORD_MAX_LENGTH_EXCEEDED_INDIVIDUAL,
                                PhraseConstraints.WORD_MAX_LENGTH)
                },
                {
                        "3 минус-слова без ключевых", "-доставка -пицца -москва",
                        new Notification(5161,
                                Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY)
                },
                {
                        "3 стоп слова без ключевых", "в из за",
                        new Notification(5141, Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY)
                },
                {
                        "Стоп-слово без ключевых", "в",
                        new Notification(5141, Api5ErrorDetailsJava.KEYWORDS_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY)
                },
                {
                        "Ключевая фраза с тремя кавычками", "\"пицца доставка\" гостиница\"",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_QUOTES_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с одной кавычкой", "\"пицца",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_QUOTES_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с точкой в конце", "пицца .",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с точкой между словами", "пицца . доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза с точкой в начале", ". пицца",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORDS)
                },
                {
                        "Ключевая фраза, содержащая невалидный символ", "aaaaa\uE751bbbbbb",
                        new Notification(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                capitalize(DeduplicateRequestItemMap.KEYWORD))
                },
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);
    }

    @Test
    public void checkDeduplicate() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withKeywords(new DeduplicateRequestItemMap().withKeyword(keyword));

        api.userSteps.keywordsResearchSteps()
                .shouldGetErrorOnDeduplicate(request, ExpectedResult.errors(notification));
    }
}
