package ru.yandex.autotests.direct.api.keywordsresearch.deduplicate;

import com.yandex.direct.api.v5.general.IdsCriteria;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateOperationEnum;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponse;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseAddItem;
import com.yandex.direct.api.v5.keywordsresearch.DeduplicateResponseUpdateItem;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.api.keywordsresearch.KeywordsResearchFeatures;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestItemMap;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.DeduplicateRequestMap;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static java.util.Collections.singletonList;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.hasSize;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assertThat;
import static ru.yandex.autotests.irt.testutils.allure.TestSteps.assumeThat;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;

@Features(KeywordsResearchFeatures.DEDUPLICATE)
@Description("Тест на общую корректность работы сервиса с операцией MERGE_DUPLICATES")
public class DeduplicateMergeDuplicatesTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);
    }

    @Test
    public void convertToAddTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(new DeduplicateRequestItemMap().withKeyword("ключевая фраза"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("ключевая фраза");
        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Добавляется корректный текст", actual, beanDiffer(expected));
    }

    @Test
    public void convertToUpdateTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -раз").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -два"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keywordUpdate = new DeduplicateResponseUpdateItem();
        keywordUpdate.setKeyword("ключевая фраза -два -раз");
        keywordUpdate.setId(1L);

        var expected = new DeduplicateResponse().withUpdate(singletonList(keywordUpdate));

        assertThat("Обновляется текст на корректный", actual, beanDiffer(expected));
    }

    @Test
    public void convertToDeleteTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -два"),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -раз").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -раз -два").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var ids = new IdsCriteria().withIds(singletonList(2L));

        var keywordUpdate = new DeduplicateResponseUpdateItem();
        keywordUpdate.setKeyword("ключевая фраза -два -раз");
        keywordUpdate.setId(1L);

        var expected = new DeduplicateResponse();
        expected.setUpdate(singletonList(keywordUpdate));
        expected.setDelete(ids);

        assertThat("Удаляется корректная фраза", actual, beanDiffer(expected));
    }

    @Test
    public void convertToFailureTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая++ фраза"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        assumeThat("Ответ Failure из сервиса получен", actual.getFailure(),
                hasSize(1));
        assertThat("Возвращается корректная позиция ошибки", actual.getFailure().get(0).getPosition(),
                equalTo(1L));
    }

    @Test
    public void samePlusWordsGlueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -раз"),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -два"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keywordAdd = new DeduplicateResponseAddItem().withKeyword("ключевая фраза -два -раз");
        var expected = new DeduplicateResponse().withAdd(singletonList(keywordAdd));

        assertThat("Ключевые фразы склеены", actual, beanDiffer(expected));
    }

    @Test
    public void samePlusWordsInKeywordGlueTest() {
        DeduplicateRequestMap request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("купить новый нова -утюг"),
                        new DeduplicateRequestItemMap().withKeyword("купить новый -пылесос"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        DeduplicateResponse expected = new DeduplicateResponse();

        DeduplicateResponseAddItem keywordAdd = new DeduplicateResponseAddItem();
        keywordAdd.setKeyword("купить новый -пылесос -утюг");

        expected.setAdd(singletonList(keywordAdd));

        assertThat("Ключевые слова склеены", actual, beanDiffer(expected));
    }

    @Test
    public void differentPlusWordsWithSameNormalizedValueGlueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                    new DeduplicateRequestItemMap().withKeyword("окна на заказ -дешево"),
                    new DeduplicateRequestItemMap().withKeyword("заказ окон -жалюзи"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("заказ окон -дешево -жалюзи");
        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Ключевые фразы склеены", actual, beanDiffer(expected));
    }

    /**
     * слово 'дешево' имеет две леммы: 'дешево' и 'дешевый',
     * <p>слово 'дешевый' имеет одну лемму: 'дешевый',
     * <p>слова имеют пересекающиеся леммы и должно остататься слово с наибольшим кол-вом лемм (т.е. слово 'дешево').
     */
    @Test
    public void differentPlusWordsWithSameLemmaGlueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("Купить дешевый холодильник -раз"),
                        new DeduplicateRequestItemMap().withKeyword("холодильник дешево купить -два"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("холодильник дешево купить -два -раз");
        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Ключевые фразы склеены", actual, beanDiffer(expected));
    }

    /**
     * слово 'ухи' может означать лишь суп (единственная лемма 'уха'),
     * <p>слово 'ухо' может означать лишь ухо (единственная лемма 'ухо'),
     * <p>леммы не пересекаются, значит склейки быть не должно.
     */
    @Test
    public void differentPlusWordsWithSameLemmaOnlyInNormalizedFormNoGlueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("Купить ухи -раз").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("ухо купить -два").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var expected = new DeduplicateResponse();

        assertThat("Ключевые фразы не должны расклеиваться", actual, beanDiffer(expected));
    }

    @Test
    public void samePlusWordsButOneQuotedGlueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("купить слона"),
                        new DeduplicateRequestItemMap().withKeyword("\"купить слона\""));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("купить слона");
        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Ключевые фразы склеены", actual, beanDiffer(expected));
    }

    @Test
    public void samePlusWordsButOneQuotedGlueReversedTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("\"купить слона\""),
                        new DeduplicateRequestItemMap().withKeyword("купить слона"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("купить слона");
        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Ключевые фразы склеены", actual, beanDiffer(expected));
    }

    @Test
    public void plusWordsInSameFormNoUnglueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -раз").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая -два").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var expected = new DeduplicateResponse();

        assertThat("Ключевые фразы не должны расклеиваться", actual, beanDiffer(expected));
    }

    @Test
    public void plusWordsInDifferentFormsNoUnglueTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевых фраз -раз").withId(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевого -два").withId(2L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var expected = new DeduplicateResponse();

        assertThat("Ключевые фразы не должны расклеиваться", actual, beanDiffer(expected));
    }

    @Test
    public void minusWordsReturnInOriginalStateTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(new DeduplicateRequestItemMap().withKeyword("куплю -слона"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("куплю -слона");

        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Минус-слова возвращаются в исходной форме", actual, beanDiffer(expected));
    }

    @Test
    public void minusWordsReturnInOriginalStateWhenGluedTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -куплю -индийского"),
                        new DeduplicateRequestItemMap().withKeyword("ключевая фраза -слона"));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var keyword = new DeduplicateResponseAddItem().withKeyword("ключевая фраза -индийского -куплю -слона");
        var expected = new DeduplicateResponse().withAdd(singletonList(keyword));

        assertThat("Минус-слова возвращаются в исходной форме после расклейки", actual,
                beanDiffer(expected));
    }

    @Test
    public void weightTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая -фраза").withId(1L).withWeight(3L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая -фраза -раз").withId(2L).withWeight(2L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая -раз").withWeight(1L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var update = new DeduplicateResponseUpdateItem();
        update.setKeyword("ключевая -раз -фраза");
        update.setId(1L);

        var ids = new IdsCriteria().withIds(singletonList(2L));

        var expected = new DeduplicateResponse();
        expected.setUpdate(singletonList(update));
        expected.setDelete(ids);

        assertThat("Веса обрабатываются корректно", actual, beanDiffer(expected));
    }

    @Test
    public void anotherWeightsWeightTest() {
        var request = new DeduplicateRequestMap()
                .withOperations(DeduplicateOperationEnum.MERGE_DUPLICATES)
                .withKeywords(
                        new DeduplicateRequestItemMap().withKeyword("ключевая -фраза").withId(1L).withWeight(1L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая -фраза -два").withId(2L).withWeight(2L),
                        new DeduplicateRequestItemMap().withKeyword("ключевая -два").withWeight(3L));

        DeduplicateResponse actual =
                api.userSteps.keywordsResearchSteps().keywordsResearchDeduplicateRawResponse(request);

        var ids = new IdsCriteria().withIds(singletonList(1L));
        var expected = new DeduplicateResponse().withDelete(ids);

        assertThat("Веса обрабатываются корректно", actual, beanDiffer(expected));
    }
}
