package ru.yandex.autotests.direct.api.keywordsresearch.hassearchvolume;

import java.util.Arrays;
import java.util.Collection;

import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.autotests.direct.api.keywordsresearch.KeywordsResearchFeatures;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetailsJava;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.HasSearchVolumeRequestMap;
import ru.yandex.autotests.directapi.model.api5.keywordsresearch.HasSearchVolumeSelectionCriteriaMap;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.autotests.directapi.steps.KeywordsResearchSteps;
import ru.yandex.direct.core.entity.keyword.service.validation.phrase.keyphrase.PhraseConstraints;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.qatools.allure.annotations.Description;
import ru.yandex.qatools.allure.annotations.Features;

import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.ALL_DEVICES;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.DESKTOPS;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.KEYWORD;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.MOBILE_PHONES;
import static com.yandex.direct.api.v5.keywordsresearch.HasSearchVolumeFieldEnum.TABLETS;
import static org.apache.commons.lang3.StringUtils.capitalize;
import static ru.yandex.autotests.directapi.model.api5.keywordsresearch.HasSearchVolumeSelectionCriteriaMap.KEYWORDS;

@Features(KeywordsResearchFeatures.HAS_SEARCH_VOLUME)
@Description("Негативные запросы с минус-словами")
@RunWith(Parameterized.class)
public class HasSearchVolumeKeywordValidationNegativeTest {
    private static final int THREE_SECONDS = 3 * 1000;
    private static final String FIRST_ELEMENT = "[0]";

    @ClassRule
    public static final ApiSteps api = new ApiSteps();
    @Rule
    public final ExpectedException thrown = ExpectedException.none();

    @Parameterized.Parameter(value = 0)
    public String description;

    @Parameterized.Parameter(value = 1)
    public String keyword;

    @Parameterized.Parameter(value = 2)
    public Notification error;

    @Parameterized.Parameters(name = "test = {0}")
    public static Collection fields() {
        Object[][] data = new Object[][] {
                {
                        "Ключевые слова содержат минус-слово", "доставка пицца -пицца",
                        new Notification(5161, Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Ключевые слова содержат минус-слово", "пицца -пицца",
                        new Notification(5161, Api5ErrorDetailsJava.MINUS_WORD_CANT_SUBTRACT_PLUS_WORDS,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Ключевое слово предваряется минус-словом", "высокий -полет квадрокоптер",
                        new Notification(5161, Api5ErrorDetailsJava.ONLY_SEPARATE_MINUS_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORD,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Минус-слово без ключевого", "-балкон",
                        new Notification(5161, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Простая группировка с минус-словами", "сервисное обслуживание (Kia | Ford) -гарантийное -бесплатное",
                        new Notification(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "2 точки в минус-слове", "пицца -57.21.45",
                        new Notification(5161, Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORD_PHRASES,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Минус-слово в кавычках", "\"-пицца\"",
                        new Notification(5161, Api5ErrorDetailsJava.BOTH_QUOTES_AND_MINUS_WORDS_IN_KEYWORD,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Неправильное использование специальных символов", "[.пицца -доставка",
                        new Notification(5002, Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS,
                                "[.пицца -доставка")
                },
                {
                        "Неправильное использование специальных символов", "пицца -'доставка -гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORD,
                                "пицца -'доставка -гостиница", "-")
                },
                {
                        "Минус-слово с точкой", "гостиница -пиц.ца -доставка",
                        new Notification(5161, Api5ErrorDetailsJava.ONLY_SEPARATE_WORDS_CAN_BE_EXCLUDED_FROM_KEYWORD_PHRASES,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Неправильное использование знака +", "пицца доставка++отель гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "пицца доставка++отель гостиница", "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка+ гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "пицца доставка+ гостиница", "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка гостиница+",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "пицца доставка гостиница+", "+")
                },
                {
                        "Неправильное использование знака +", "пицца доставка+гостиница отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "пицца доставка+гостиница отель", "+")
                },
                {
                        "Неправильное использование знака -", "пицца отель--гостиница доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORD,
                                "пицца отель--гостиница доставка", "-")
                },
                {
                        "Неправильное использование знака -", "пицца доставка- гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORD,
                                "пицца доставка- гостиница", "-")
                },
                {
                        "Неправильное использование знака -", "пицца доставка отель-",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_MINUS_SIGN_IN_KEYWORD,
                                "пицца доставка отель-", "-")
                },
                {
                        "Неправильное использование знака !", "пицца отель !-доставка гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORD,
                                "пицца отель !-доставка гостиница", "!")
                },
                {
                        "Неправильное использование знака !", "пицца отель X-!доставка гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORD,
                                "пицца отель X-!доставка гостиница", "!")
                },
                {
                        "Неправильное использование знака !", "пицца доставка -!-отель гостиница",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORD,
                                "пицца доставка -!-отель гостиница", "!")
                },
                {
                        "Неправильное использование знака +", "гостиница отель +-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "гостиница отель +-пицца доставка", "+")
                },
                {
                        "Неправильное использование знака +", "гостиница отель X-+пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "гостиница отель X-+пицца доставка", "+")
                },
                {
                        "Неправильное использование знака +", "гостиница отель -+-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "гостиница отель -+-пицца доставка", "+")
                },
                {
                        "Восклицательный знак с другим модификатором", "гостиница !\"отель-пицца\" доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORD,
                                "гостиница !\"отель-пицца\" доставка", "!")
                },
                {
                        "Восклицательный знак в конце фразы", "гостиница отель-пицца доставка!",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORD,
                                "гостиница отель-пицца доставка!", "!")
                },
                {
                        "Восклицательный знак без слова", "гостиница ! отель-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORD,
                                "гостиница ! отель-пицца доставка", "!")
                },
                {
                        "Два восклицательных знака подряд", "гостиница !!отель-пицца доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_EXCLAMATION_MARKS_IN_KEYWORD,
                                "гостиница !!отель-пицца доставка", "!")
                },
                {
                        "Знак минус внутри квадратных скобок", "пицца [-отель] доставка",
                        new Notification(5002, Api5ErrorDetailsJava.NO_MODIFIERS_INSIDE_SQUARE_BRACKETS,
                                "пицца [-отель] доставка")
                },
                {
                        "Модификаторы внутри квадратных скобок", "пицца [+\"отель\"] доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_PLUS_SIGN_IN_KEYWORD,
                                "пицца [+\"отель\"] доставка", "+")
                },
                {
                        "Вложенные квадратные скобки", "пицца [ [отель] доставка]",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [ [отель] доставка]")
                },
                {
                        "Пустые квадратные скобки", "пицца [] отель доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [] отель доставка")
                },
                {
                        "Квадратные скобки в неправильном порядке", "]пицца[",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "]пицца[")
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [отель [доставка гостиница]",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [отель [доставка гостиница]")
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца гостиница] отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца гостиница] отель")
                },
                {
                        "Неправильная скобочная последовательность для квадратных скобок", "пицца [гостиница отель",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_BRACKETS_IN_KEYWORD,
                                "пицца [гостиница отель")
                },
                {
                        "8 ключевых слов", "пицца отель москва гостиница поезд самолет такси теплоход",
                        new Notification(5140, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONTAIN_MORE_WORDS,
                                capitalize(KEYWORDS) + FIRST_ELEMENT, PhraseConstraints.WORDS_MAX_COUNT)
                },
                {
                        "Слишком длинное ключевое слово", "отель пииииииииииииииииииииииииииииииииииииииииииицца теплоход",
                        new Notification(5142, Api5ErrorDetailsJava.KEYWORD_MAX_LENGTH_EXCEEDED_INDIVIDUAL,
                                PhraseConstraints.WORD_MAX_LENGTH)
                },
                {
                        "3 минус-слова без ключевых", "-доставка -пицца -москва",
                        new Notification(5161, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_NEGATIVE_KEYWORDS_ONLY,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "3 стоп слова без ключевых", "в из за",
                        new Notification(5141, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Стоп-слово без ключевых", "в",
                        new Notification(5141, Api5ErrorDetailsJava.KEYWORD_COULD_NOT_CONSIST_OF_STOPWORDS_ONLY,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Ключевая фраза с тремя кавычками", "\"пицца доставка\" гостиница\"",
                        new Notification(5002, Api5ErrorDetailsJava.KEYWORD_INCORRECT_USAGE_OF_QUOTES,
                                "\"пицца доставка\" гостиница\"")
                },
                {
                        "Ключевая фраза с одной кавычкой", "\"пицца",
                        new Notification(5002, Api5ErrorDetailsJava.KEYWORD_INCORRECT_USAGE_OF_QUOTES,
                                "\"пицца")
                },
                {
                        "Ключевая фраза с точкой в конце", "пицца .",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORD,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Ключевая фраза с точкой между словами", "пицца . доставка",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORD,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Ключевая фраза с точкой в начале", ". пицца",
                        new Notification(5002, Api5ErrorDetailsJava.INCORRECT_USE_OF_DOT_IN_KEYWORD,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
                {
                        "Ключевая фраза, содержащая невалидный символ", "aaaaa\uE751bbbbbb",
                        new Notification(5002, Api5ErrorDetailsJava.INVALID_CHARS,
                                capitalize(KEYWORDS) + FIRST_ELEMENT)
                },
        };
        return Arrays.asList(data);
    }

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);
    }

    @Before
    public void delayRequest() throws InterruptedException {
        // Разрешено не более 20 запросов за окно в 60 секунд
        Thread.sleep(THREE_SECONDS);
    }

    @Test
    public void checkHasSearchVolume() {
        HasSearchVolumeRequestMap request = new HasSearchVolumeRequestMap()
                .withSelectionCriteria(new HasSearchVolumeSelectionCriteriaMap()
                        .withKeywords(keyword)
                        .withRegionIds(0L))
                .withFieldNames(KEYWORD, DESKTOPS, MOBILE_PHONES, TABLETS, ALL_DEVICES);

        KeywordsResearchSteps keywordsResearchSteps = api.userSteps.keywordsResearchSteps();
        thrown.expect(keywordsResearchSteps.getApi5JsonErrorMatcher(error));
        keywordsResearchSteps.keywordsResearchHasSearchVolumeRawResponse(request);
    }
}
