package ru.yandex.autotests.direct.api.smartadtargets;

import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.smartadtargets.AddRequest;
import com.yandex.direct.api.v5.smartadtargets.AddResponse;
import com.yandex.direct.api.v5.smartadtargets.ConditionsArray;
import com.yandex.direct.api.v5.smartadtargets.SmartAdTargetGetItem;
import org.assertj.core.api.SoftAssertions;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.utils.differs.JAXBElementDiffer;
import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonError;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.autotests.directapi.steps.PerformanceFilterSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.CompareStrategy;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.info.PerformanceFilterInfo;
import ru.yandex.qatools.allure.annotations.Features;

import static ru.yandex.autotests.direct.api.smartadtargets.TestSmartAdTargets.createDefaultGetItemWithoutConditions;
import static ru.yandex.autotests.direct.api.smartadtargets.TestSmartAdTargets.defaultConditions;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;
import static ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies.onlyExpectedFields;
import static ru.yandex.direct.test.utils.assertj.Conditions.matchedBy;

@Features(SmartAdTargetsFeatures.ADD)
public class AddSmartAdTargetsTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();
    private static PerformanceFilterSteps performanceFilterSteps;

    private static Long adGroupId;
    private static Long campaignId;

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);

        performanceFilterSteps = api.userSteps().performanceFilterSteps();
        PerformanceFilterInfo startFilterInfo = performanceFilterSteps.addDefaultPerformanceFilter(clientInfo);
        adGroupId = startFilterInfo.getAdGroupId();
        campaignId = startFilterInfo.getCampaignId();
    }

    @Test
    public void addSmartAdTarget_success() {
        AddRequest addRequest = TestSmartAdTargets.createDefaultAddRequest(adGroupId);
        AddResponse addResponse = performanceFilterSteps.shouldGetResultOnAddRequest(
                addRequest,
                ExpectedResult.success());

        Long addedFilterId = addResponse.getAddResults().get(0).getId();
        SmartAdTargetGetItem actualItem = performanceFilterSteps.getSmartAdTargetGetItem(addedFilterId);
        ConditionsArray actualConditions = actualItem.getConditions().getValue();

        SmartAdTargetGetItem expectedItem = createDefaultGetItemWithoutConditions(addedFilterId, adGroupId, campaignId);
        ConditionsArray expectedConditions = defaultConditions();

        CompareStrategy compareStrategy = onlyExpectedFields()
                .forClasses(JAXBElement.class)
                .useDiffer(new JAXBElementDiffer());

        SoftAssertions.assertSoftly(soft -> {
            soft.assertThat(actualItem).as("The added item is the same as expected")
                    .is(matchedBy(beanDiffer(expectedItem).useCompareStrategy(compareStrategy)));
            soft.assertThat(actualConditions).as("The added conditions is the same as expected")
                    .is(matchedBy(beanDiffer(expectedConditions).useCompareStrategy(onlyExpectedFields())));
        });
    }

    @Test
    public void addSmartAdTarget_withWrongPriceCpa_failure() {
        int tooBigPrice = Integer.MAX_VALUE;
        AddRequest addRequest = TestSmartAdTargets.createDefaultAddRequest(adGroupId);
        addRequest.getSmartAdTargets()
                .get(0).withAverageCpa(tooBigPrice * 1_000_000L);

        performanceFilterSteps.shouldGetResultOnAddRequest(
                addRequest,
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.INVALID_VALUE_SHORT, "AverageCpa", tooBigPrice)));
    }

    @Test(expected = Api5JsonError.class)
    public void addSmartAdTarget_withEmptyRequest_failure() {
        AddRequest addRequest = new AddRequest();
        performanceFilterSteps.sendAddRequest(addRequest);
    }

}
