package ru.yandex.autotests.direct.api.smartadtargets;

import java.util.Arrays;
import java.util.List;

import com.yandex.direct.api.v5.smartadtargets.AddRequest;
import com.yandex.direct.api.v5.smartadtargets.AddResponse;
import com.yandex.direct.api.v5.smartadtargets.DeleteRequest;
import com.yandex.direct.api.v5.smartadtargets.DeleteResponse;
import com.yandex.direct.api.v5.smartadtargets.ResumeRequest;
import com.yandex.direct.api.v5.smartadtargets.ResumeResponse;
import com.yandex.direct.api.v5.smartadtargets.SetBidsRequest;
import com.yandex.direct.api.v5.smartadtargets.SetBidsResponse;
import com.yandex.direct.api.v5.smartadtargets.SuspendRequest;
import com.yandex.direct.api.v5.smartadtargets.SuspendResponse;
import com.yandex.direct.api.v5.smartadtargets.UpdateRequest;
import com.yandex.direct.api.v5.smartadtargets.UpdateResponse;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.autotests.directapi.apiclient.ApiClient;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.direct.core.testing.data.TestRoleRelation;
import ru.yandex.direct.core.testing.info.PerformanceAdGroupInfo;
import ru.yandex.direct.core.testing.info.PerformanceFilterInfo;
import ru.yandex.direct.core.testing.info.RoleRelationInfo;
import ru.yandex.direct.core.testing.info.UserInfo;
import ru.yandex.qatools.allure.annotations.Features;

import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.autotests.directapi.model.api5.Action.ADD;
import static ru.yandex.autotests.directapi.model.api5.Action.DELETE;
import static ru.yandex.autotests.directapi.model.api5.Action.RESUME;
import static ru.yandex.autotests.directapi.model.api5.Action.SET_BIDS;
import static ru.yandex.autotests.directapi.model.api5.Action.SUSPEND;
import static ru.yandex.autotests.directapi.model.api5.Action.UPDATE;
import static ru.yandex.autotests.directapi.model.api5.ServiceNames.SMART_AD_TARGETS;

@Features({SmartAdTargetsFeatures.ADD, SmartAdTargetsFeatures.UPDATE, SmartAdTargetsFeatures.DELETE,
        SmartAdTargetsFeatures.RESUME, SmartAdTargetsFeatures.SET_BIDS, SmartAdTargetsFeatures.SUSPEND})
@RunWith(Parameterized.class)
public class ChangeSmartAdTargetsByRolesTest {

    @Parameterized.Parameter()
    public String description;

    @Parameterized.Parameter(1)
    public TestRoleRelation roleRelation;

    @Parameterized.Parameter(2)
    public Object expectedResult;

    private ApiSteps api;

    @Parameterized.Parameters(name = "{0}")
    public static List<Object[]> testData() {
        Object[][] data = new Object[][]{
                {"Тот же клиент", TestRoleRelation.CLIENT_CHIEF_AND_SAME_CLIENT, "size=1"},
                {"Посторонний клиент", TestRoleRelation.CLIENT_AND_OTHER_CLIENT, "Объект не найден"},
                {"Агенство и клиент", TestRoleRelation.AGENCY_AND_SUB_CLIENT, "size=1"},
                {"IDM менеджер и клиент", TestRoleRelation.IDM_MANAGER_AND_CLIENT, "size=1"},
                {"Менеджер и его клиент", TestRoleRelation.CLIENT_MANAGER_AND_CLIENT, "size=1"},
                {"Представитель клиента", TestRoleRelation.REP_AND_CHIEF, "size=1"},
                {"Суперпользователь", TestRoleRelation.SUPER_AND_CLIENT, "size=1"},
                {"Суперриадер", TestRoleRelation.SUPER_READER_AND_CLIENT, "Нет прав"},
                {"Саппорт", TestRoleRelation.SUPPORT_AND_CLIENT, "size=1"},
        };
        return Arrays.asList(data);
    }

    @Before
    public void before() {
        api = new ApiSteps();
    }

    @Test
    public void add() {
        RoleRelationInfo roleRelationInfo = api.coreSteps().rolesSteps().getRoleRelationInfo(roleRelation);
        PerformanceAdGroupInfo adGroupInfo = getAdGroupInfo(api, roleRelationInfo);
        AddRequest request = TestSmartAdTargets.createDefaultAddRequest(adGroupInfo.getAdGroupId());
        String clientLogin = roleRelationInfo.getOwnerUserInfo().getLogin();
        ApiClient apiClient = getApiClientWithOperator(roleRelationInfo.getOperatorUserInfo());

        Object result;
        try {
            AddResponse response = apiClient.invokeMethod(SMART_AD_TARGETS, clientLogin, ADD, request);
            result = "size=" + response.getAddResults().size();
        } catch (Exception ex) {
            result = ex.getMessage();
        }
        assertThat(result).as("Результат").isEqualTo(expectedResult);
    }

    @Test
    public void delete() {
        RoleRelationInfo roleRelationInfo = api.coreSteps().rolesSteps().getRoleRelationInfo(roleRelation);
        Long filterId = createSmartTarget(roleRelationInfo);
        DeleteRequest request = TestSmartAdTargets.createDeleteRequest(filterId);
        String clientLogin = roleRelationInfo.getOwnerUserInfo().getLogin();
        ApiClient apiClient = getApiClientWithOperator(roleRelationInfo.getOperatorUserInfo());

        Object result;
        try {
            DeleteResponse response = apiClient.invokeMethod(SMART_AD_TARGETS, clientLogin, DELETE, request);
            result = "size=" + response.getDeleteResults().size();
        } catch (Exception ex) {
            result = ex.getMessage();
        }
        assertThat(result).as("Результат").isEqualTo(expectedResult);
    }

    @Test
    public void update() {
        RoleRelationInfo roleRelationInfo = api.coreSteps().rolesSteps().getRoleRelationInfo(roleRelation);
        Long filterId = createSmartTarget(roleRelationInfo);
        UpdateRequest request = TestSmartAdTargets.createDefaultUpdateRequest(filterId);
        String clientLogin = roleRelationInfo.getOwnerUserInfo().getLogin();
        ApiClient apiClient = getApiClientWithOperator(roleRelationInfo.getOperatorUserInfo());

        Object result;
        try {
            UpdateResponse response = apiClient.invokeMethod(SMART_AD_TARGETS, clientLogin, UPDATE, request);
            result = "size=" + response.getUpdateResults().size();
        } catch (Exception ex) {
            result = ex.getMessage();
        }
        assertThat(result).as("Результат").isEqualTo(expectedResult);
    }

    @Test
    public void resume() {
        RoleRelationInfo roleRelationInfo = api.coreSteps().rolesSteps().getRoleRelationInfo(roleRelation);
        Long filterId = createSmartTarget(roleRelationInfo);
        ResumeRequest request = TestSmartAdTargets.createResumeRequest(filterId);
        String clientLogin = roleRelationInfo.getOwnerUserInfo().getLogin();
        ApiClient apiClient = getApiClientWithOperator(roleRelationInfo.getOperatorUserInfo());

        Object result;
        try {
            ResumeResponse response = apiClient.invokeMethod(SMART_AD_TARGETS, clientLogin, RESUME, request);
            result = "size=" + response.getResumeResults().size();
        } catch (Exception ex) {
            result = ex.getMessage();
        }
        assertThat(result).as("Результат").isEqualTo(expectedResult);
    }

    @Test
    public void suspend() {
        RoleRelationInfo roleRelationInfo = api.coreSteps().rolesSteps().getRoleRelationInfo(roleRelation);
        Long filterId = createSmartTarget(roleRelationInfo);
        SuspendRequest request = TestSmartAdTargets.createSuspendRequest(filterId);
        String clientLogin = roleRelationInfo.getOwnerUserInfo().getLogin();
        ApiClient apiClient = getApiClientWithOperator(roleRelationInfo.getOperatorUserInfo());

        Object result;
        try {
            SuspendResponse response = apiClient.invokeMethod(SMART_AD_TARGETS, clientLogin, SUSPEND, request);
            result = "size=" + response.getSuspendResults().size();
        } catch (Exception ex) {
            result = ex.getMessage();
        }
        assertThat(result).as("Результат").isEqualTo(expectedResult);
    }

    @Test
    public void setBids() {
        RoleRelationInfo roleRelationInfo = api.coreSteps().rolesSteps().getRoleRelationInfo(roleRelation);
        Long filterId = createSmartTarget(roleRelationInfo);
        SetBidsRequest request = TestSmartAdTargets.createDefaultSetBidsRequest(filterId);
        String clientLogin = roleRelationInfo.getOwnerUserInfo().getLogin();
        ApiClient apiClient = getApiClientWithOperator(roleRelationInfo.getOperatorUserInfo());

        Object result;
        try {
            SetBidsResponse response = apiClient.invokeMethod(SMART_AD_TARGETS, clientLogin, SET_BIDS, request);
            result = "size=" + response.getSetBidsResults().size();
        } catch (Exception ex) {
            result = ex.getMessage();
        }
        assertThat(result).as("Результат").isEqualTo(expectedResult);
    }

    private Long createSmartTarget(RoleRelationInfo roleRelationInfo) {
        PerformanceAdGroupInfo adGroupInfo = getAdGroupInfo(api, roleRelationInfo);
        PerformanceFilterInfo filterInfo =
                api.coreSteps().performanceFilterSteps()
                        .addPerformanceFilter(new PerformanceFilterInfo().withAdGroupInfo(adGroupInfo));
        return filterInfo.getFilterId();
    }

    private ApiClient getApiClientWithOperator(UserInfo operatorUserInfo) {
        api.as(operatorUserInfo.getUser());
        return api.userSteps().performanceFilterSteps().defaultClientV5();
    }

    private PerformanceAdGroupInfo getAdGroupInfo(ApiSteps api, RoleRelationInfo roleRelationInfo) {
        PerformanceAdGroupInfo adGroupInfo =
                api.coreSteps().adGroupSteps().createDefaultPerformanceAdGroup(roleRelationInfo.getOwnerClientInfo());
        Long operatorUid = roleRelationInfo.getOperatorUserInfo().getUid();
        return adGroupInfo;
    }

}
