package ru.yandex.autotests.direct.api.smartadtargets;

import javax.xml.bind.JAXBElement;

import com.yandex.direct.api.v5.smartadtargets.AddRequest;
import com.yandex.direct.api.v5.smartadtargets.AddResponse;
import com.yandex.direct.api.v5.smartadtargets.ConditionsArray;
import com.yandex.direct.api.v5.smartadtargets.GetRequest;
import com.yandex.direct.api.v5.smartadtargets.GetResponse;
import com.yandex.direct.api.v5.smartadtargets.SmartAdTargetGetItem;
import org.assertj.core.api.SoftAssertions;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.direct.utils.differs.JAXBElementDiffer;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.autotests.directapi.steps.PerformanceFilterSteps;
import ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.CompareStrategy;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.info.PerformanceFilterInfo;
import ru.yandex.qatools.allure.annotations.Features;

import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.autotests.direct.api.smartadtargets.TestSmartAdTargets.createDefaultGetItemWithoutConditions;
import static ru.yandex.autotests.direct.api.smartadtargets.TestSmartAdTargets.defaultConditions;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;
import static ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies.onlyExpectedFields;
import static ru.yandex.direct.test.utils.assertj.Conditions.matchedBy;

@Features(SmartAdTargetsFeatures.GET)
public class GetSmartAdTargetsTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();

    private static PerformanceFilterSteps performanceFilterSteps;

    private static Long adGroupId;
    private static Long campaignId;

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);

        performanceFilterSteps = api.userSteps().performanceFilterSteps();
        PerformanceFilterInfo startFilterInfo = performanceFilterSteps.addDefaultPerformanceFilter(clientInfo);
        adGroupId = startFilterInfo.getAdGroupId();
        campaignId = startFilterInfo.getCampaignId();
    }

    @Test
    public void getSmartAdTarget_success() {
        AddRequest addRequest = TestSmartAdTargets.createDefaultAddRequest(adGroupId);
        AddResponse addResponse = performanceFilterSteps.shouldGetResultOnAddRequest(addRequest,
                ExpectedResult.success());
        Long perfFilterId = addResponse.getAddResults().get(0).getId();

        GetRequest getRequest = TestSmartAdTargets.createGetRequest(perfFilterId);
        GetResponse getResponse = performanceFilterSteps.sendGetRequest(getRequest);
        SmartAdTargetGetItem actualItem = getResponse.getSmartAdTargets().get(0);
        ConditionsArray actualConditions = actualItem.getConditions().getValue();

        SmartAdTargetGetItem expectedItem = createDefaultGetItemWithoutConditions(perfFilterId, adGroupId, campaignId);
        ConditionsArray expectedConditions = defaultConditions();

        CompareStrategy compareStrategy = onlyExpectedFields()
                .forClasses(JAXBElement.class)
                .useDiffer(new JAXBElementDiffer());
        SoftAssertions.assertSoftly(soft -> {
            soft.assertThat(actualItem).as("The received item is the same as expected")
                    .is(matchedBy(beanDiffer(expectedItem).useCompareStrategy(compareStrategy)));
            soft.assertThat(actualConditions).as("The received answer conditions is the same as expected")
                    .is(matchedBy(beanDiffer(expectedConditions).useCompareStrategy(onlyExpectedFields())));
        });
    }

    @Test
    public void getSmartAdTarget_whenPerfFilterNotExist_emptyResult() {
        long notExistPerfFilterId = Integer.MAX_VALUE - 1L;
        GetRequest getRequest = TestSmartAdTargets.createGetRequest(notExistPerfFilterId);
        GetResponse getResponse = performanceFilterSteps.sendGetRequest(getRequest);
        assertThat(getResponse.getSmartAdTargets()).isEmpty();
    }

    @Test
    public void getSmartAdTarget_whenPerfFilterOwnSomebodyElse_emptyResult() {
        PerformanceFilterInfo somebodyElseFilterInfo = performanceFilterSteps.createDefaultPerformanceFilter();
        GetRequest getRequest = TestSmartAdTargets.createGetRequest(somebodyElseFilterInfo.getFilterId());
        GetResponse getResponse = performanceFilterSteps.sendGetRequest(getRequest);
        assertThat(getResponse.getSmartAdTargets()).isEmpty();
    }

}
