package ru.yandex.autotests.direct.api.smartadtargets;

import com.yandex.direct.api.v5.general.YesNoEnum;
import com.yandex.direct.api.v5.smartadtargets.ConditionsArray;
import com.yandex.direct.api.v5.smartadtargets.SmartAdTargetGetItem;
import com.yandex.direct.api.v5.smartadtargets.UpdateRequest;
import org.assertj.core.api.SoftAssertions;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import ru.yandex.autotests.directapi.apiclient.errors.Api5ErrorDetails;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonError;
import ru.yandex.autotests.directapi.model.api5.general.ExpectedResult;
import ru.yandex.autotests.directapi.model.api5.general.Notification;
import ru.yandex.autotests.directapi.steps.ApiSteps;
import ru.yandex.autotests.directapi.steps.PerformanceFilterSteps;
import ru.yandex.direct.core.testing.info.ClientInfo;
import ru.yandex.direct.core.testing.info.PerformanceFilterInfo;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.qatools.allure.annotations.Features;

import static ru.yandex.autotests.direct.api.smartadtargets.TestSmartAdTargets.DEFAULT_NEW_CPA_PRICE;
import static ru.yandex.autotests.direct.api.smartadtargets.TestSmartAdTargets.otherConditions;
import static ru.yandex.autotests.irt.testutils.beandiffer2.BeanDifferMatcher.beanDiffer;
import static ru.yandex.autotests.irt.testutils.beandiffer2.comparestrategy.defaultcomparestrategy.DefaultCompareStrategies.onlyExpectedFields;
import static ru.yandex.direct.api.v5.entity.smartadtargets.converter.CommonConverters.FACTORY;
import static ru.yandex.direct.test.utils.assertj.Conditions.matchedBy;

@Features(SmartAdTargetsFeatures.UPDATE)
public class UpdateSmartAdTargetsTest {

    @ClassRule
    public static ApiSteps api = new ApiSteps();
    private static PerformanceFilterSteps performanceFilterSteps;
    private static Long perfFilterId;
    private static ClientId clientId;

    @BeforeClass
    public static void init() {
        ClientInfo clientInfo = api.createClient();
        api.as(clientInfo);
        clientId = clientInfo.getClientId();

        performanceFilterSteps = api.userSteps().performanceFilterSteps();
        PerformanceFilterInfo startFilterInfo = performanceFilterSteps.addDefaultPerformanceFilter(clientInfo);
        perfFilterId = startFilterInfo.getFilterId();
    }

    @Test
    public void updateSmartAdTarget_success() {
        UpdateRequest updateRequest = TestSmartAdTargets.createDefaultUpdateRequest(perfFilterId);
        performanceFilterSteps.shouldGetResultOnUpdateRequest(updateRequest, ExpectedResult.success(perfFilterId));

        SmartAdTargetGetItem actualItem = performanceFilterSteps.getSmartAdTargetGetItem(perfFilterId);
        long expectedCpaPrice = DEFAULT_NEW_CPA_PRICE * 1_000_000L;
        SoftAssertions.assertSoftly(soft -> {
            soft.assertThat(actualItem.getName()).as("Updated name")
                    .isEqualTo(TestSmartAdTargets.DEFAULT_NEW_NAME);
            soft.assertThat(actualItem.getAverageCpa().getValue()).as("Updated Cpa price")
                    .isEqualTo(expectedCpaPrice);
        });
    }

    @Test
    public void updateSmartAdTarget_conditions_success() {
        api.userSteps().featureSteps().setClientFeature(clientId, FeatureName.UPDATE_FILTER_CONDITIONS_ALLOWED, true);

        UpdateRequest updateRequest = TestSmartAdTargets.createConditionsUpdateRequest(perfFilterId);
        performanceFilterSteps.shouldGetResultOnUpdateRequest(updateRequest, ExpectedResult.success(perfFilterId));

        SmartAdTargetGetItem actualItem = performanceFilterSteps.getSmartAdTargetGetItem(perfFilterId);
        YesNoEnum availableItemsOnly = actualItem.getAvailableItemsOnly().getValue();
        ConditionsArray actualConditions = actualItem.getConditions().getValue();

        ConditionsArray expectedConditions = otherConditions();
        SoftAssertions.assertSoftly(soft -> {
            soft.assertThat(availableItemsOnly).as("AvailableItemsOnly")
                    .isEqualTo(YesNoEnum.NO);
            soft.assertThat(actualConditions).as("Conditions")
                    .is(matchedBy(beanDiffer(expectedConditions).useCompareStrategy(onlyExpectedFields())));
        });
    }

    @Test
    public void updateSmartAdTarget_withWrongPriceCpa_failure() {
        int tooBigPrice = Integer.MAX_VALUE;
        UpdateRequest updateRequest = TestSmartAdTargets.createDefaultUpdateRequest(perfFilterId);
        updateRequest.getSmartAdTargets()
                .get(0)
                .withAverageCpa(FACTORY.createSmartAdTargetUpdateItemAverageCpa(tooBigPrice * 1_000_000L));
        performanceFilterSteps.shouldGetResultOnUpdateRequest(
                updateRequest,
                ExpectedResult.errors(
                        new Notification(5005, Api5ErrorDetails.INVALID_VALUE_SHORT, "AverageCpa", tooBigPrice)));
    }

    @Test(expected = Api5JsonError.class)
    public void updateSmartAdTarget_withEmptyRequest_failure() {
        UpdateRequest updateRequest = new UpdateRequest();
        performanceFilterSteps.sendUpdateRequest(updateRequest);
    }

}
