package ru.yandex.autotests.direct.utils.matchers;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.hamcrest.Description;
import org.hamcrest.Factory;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

import static org.apache.commons.lang.StringUtils.capitalize;
import static org.hamcrest.CoreMatchers.equalTo;

public class BeanEquals<T> extends TypeSafeDiagnosingMatcher<T> {
    private final T expectedBean;
    private Set<String> shouldCheckFields;
    private BeanCompareStrategy beanCompareStrategy;

    public BeanEquals(T expectedBean) {
        this.expectedBean = expectedBean;
        beanCompareStrategy = new BeanCompareStrategy();
        findOutFieldsToCheck();
    }

    /**
     * Creates a matcher that matches when the examined object has values for all of
     * its JavaBean properties (except arrays) that are equal to the corresponding values of the
     * specified bean
     * <p/>
     * For example:
     * <pre>assertThat(myBean, beanEquals(myExpectedBean))</pre>
     *
     * @param expectedBean the bean against which examined beans are compared
     */
    @Factory
    public static <T> BeanEquals<T> beanEquals(T expectedBean) {
        return new BeanEquals<>(expectedBean);
    }

    public static boolean isSimpleField(Class clazz) {
        return clazz.equals(String.class) || clazz.isEnum() || clazz.equals(Boolean.class) ||
                clazz.isPrimitive() || Number.class.isAssignableFrom(clazz);
    }

    public static Object getProperty(Object bean, String name) {
        Method method;
        try {
            method = new PropertyDescriptor(name, bean.getClass()).getReadMethod();
            return method.invoke(bean);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public BeanEquals<T> accordingStrategy(BeanCompareStrategy beanCompareStrategy) {
        this.beanCompareStrategy = beanCompareStrategy;
        for (String strategyFields : beanCompareStrategy.getFieldNames()) {
            shouldCheckFields.add(strategyFields);
        }
        return this;
    }

    /**
     * Only specified fields will be take into account on beans comparison, all the other should be ignored
     *
     * @param fieldNames
     * @return
     */
    public BeanEquals<T> byFields(String... fieldNames) {
        shouldCheckFields.retainAll(Arrays.asList(fieldNames));
        return this;
    }

    /**
     * Let mather know fields that should be ignored in beans comparison
     * Field names should start with only small letter
     *
     * @param fieldNames
     * @return
     */
    public BeanEquals<T> ignoreFields(String... fieldNames) {
        shouldCheckFields.removeAll(Arrays.asList(fieldNames));
        return this;
    }

    private void findOutFieldsToCheck() {
        shouldCheckFields = new HashSet<>();
        for (var property : PropertyUtils.getPropertyDescriptors(expectedBean)) {
            if (property.getReadMethod() != null && isSimpleField(property.getPropertyType())) {
                shouldCheckFields.add(property.getName());
            }
        }
    }

    private Matcher getMatcherForField(String fieldName) {
        Matcher matcher = beanCompareStrategy.getFieldMatcher(fieldName);
        try {
            if (matcher == null) {
                matcher = equalTo(getProperty(expectedBean, fieldName));
                beanCompareStrategy.putFieldMatcher(fieldName, matcher);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return matcher;
    }

    @Override
    public void describeTo(Description description) {
        description.appendText("bean with properties {");
        try {
            for (String fieldName : shouldCheckFields) {
                Object expectedValue = getProperty(expectedBean, fieldName);
                if (expectedValue == null && beanCompareStrategy.getFieldMatcher(fieldName) == null) {
                    continue;
                }
                description.appendText("\n\t").
                        appendText(capitalize(fieldName) + ": ");
                getMatcherForField(fieldName).describeTo(description);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        description.appendText("\n}");
    }

    @Override
    protected boolean matchesSafely(T actualBean, Description mismatchDescription) {
        boolean matchesResult = true;
        try {
            for (String fieldName : shouldCheckFields) {
                Object expectedValue = getProperty(expectedBean, fieldName);
                if (expectedValue == null && beanCompareStrategy.getFieldMatcher(fieldName) == null) {
                    continue;
                }
                Object actualValue = getProperty(actualBean, fieldName);
                if (!getMatcherForField(fieldName).matches(actualValue)) {
                    matchesResult = false;
                    mismatchDescription.appendText("[" + capitalize(fieldName) + "] ");
                    getMatcherForField(fieldName).describeMismatch(actualValue, mismatchDescription);
                    mismatchDescription.appendText("\n\t\t  ");
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return matchesResult;
    }

    public T getExpectedBean() {
        return expectedBean;
    }
}

