package ru.yandex.autotests.directapi.apiclient;

import java.io.StringWriter;
import java.io.Writer;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import com.google.gson.JsonSyntaxException;
import freemarker.cache.ClassTemplateLoader;
import freemarker.cache.TemplateLoader;
import freemarker.template.Configuration;
import freemarker.template.DefaultObjectWrapper;
import freemarker.template.Template;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHeader;

import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonError;
import ru.yandex.autotests.directapi.apiclient.errors.RbacErrorException;
import ru.yandex.autotests.directapi.apiclient.internal.JsonUtils;
import ru.yandex.autotests.directapi.apiclient.internal.MethodInvocationResult;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.ServiceNames;

import static freemarker.template.Configuration.VERSION_2_3_0;

public abstract class ApiClient {
    private static final Pattern RBAC_ERROR_RE =
            Pattern.compile("^rbac_move_nscamp_to_scamp\\(\\$rbac, \\d+, \\d+, \\d+\\) == 990001.*$",
                    Pattern.MULTILINE | Pattern.DOTALL);
    public static Log log = LogFactory.getLog(ApiClient.class);

    static {
        disableSslVerification();
    }

    protected ConnectionConfig connectionConfig;
    protected RequestHeader requestHeader;

    public ApiClient(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        this.connectionConfig = connectionConfig;
        this.requestHeader = requestHeader;
    }
    //TODO: move to utils

    public static ContentType getHttpResponseContentType(String url) {
        try {
            CloseableHttpClient httpClient = HttpClientFactory.getHttpClient();
            HttpGet httpRequest = new HttpGet(url);
            CloseableHttpResponse response = httpClient.execute(httpRequest);
            try {
                return ContentType.parse(response.getEntity().getContentType().getValue());
            } finally {
                response.close();
            }
        } catch (Exception e) {
            throw new DirectAPIException("Ошибка обработки доступности URL", e);
        }

    }

    public static void disableSslVerification() {
        try {
            // Create a trust manager that does not validate certificate chains
            TrustManager[] trustAllCerts = new TrustManager[]{new X509TrustManager() {
                public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                    return null;
                }

                public void checkClientTrusted(X509Certificate[] certs, String authType) {
                }

                public void checkServerTrusted(X509Certificate[] certs, String authType) {
                }
            }
            };

            // Install the all-trusting trust manager
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());

            // Create all-trusting host name verifier
            HostnameVerifier allHostsValid = new HostnameVerifier() {
                public boolean verify(String hostname, SSLSession session) {
                    return true;
                }
            };

            // Install the all-trusting host verifier
            HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (KeyManagementException e) {
            e.printStackTrace();
        }
    }

    public String login() {
        return requestHeader.getLogin();
    }

    /**
     * @param serviceName
     * @param login       subclient or serviced client login used to pass to Client-Login Header
     * @param methodName
     * @param params
     * @param <T>
     * @return
     */
    public final <T> T invokeMethod(ServiceNames serviceName, String login, Action methodName, Object params) {
        return this.<T>invokeMethodEx(serviceName, login, methodName, params).getResponseObject();
    }

    protected abstract ProtocolType getProtocolType();

    public abstract <T> MethodInvocationResult<T> invokeMethodEx(
            ServiceNames serviceName, String login, Action methodName, Object params);

    protected Map<String, Object> getHttpHeaders(String clientLogin) {
        Map<String, Object> requestHeaders = new HashMap<>();
        if (requestHeader.getToken() != null) {
            requestHeaders.put("Authorization", "Bearer " + requestHeader.getToken());
        }
        if (clientLogin != null) {
            requestHeaders.put("Client-Login", clientLogin);
        }
        requestHeaders.put("Accept-Language", requestHeader.getLocale());
        return requestHeaders;
    }

    protected Header[] toHeadersArray(Map<String, Object> headersMap) {
        int index = 0;
        Header[] headers = new Header[headersMap.size()];
        for (String key : headersMap.keySet()) {
            headers[index++] = new BasicHeader(key, headersMap.get(key).toString());
        }
        return headers;
    }

    /**
     * Look http message for error codes and throw AxisError if found one
     *
     * @param response
     */
    void checkJsonFaultPresence(String response, boolean isJavaResponse) {
        try {
            Api5JsonError error = JsonUtils.getObject(response, Api5JsonError.class);
            if (error.getErrorCode() != null) {
                error.setJavaResponse(isJavaResponse);
                if (error.getErrorCode() == 1002 && RBAC_ERROR_RE.matcher(error.getErrorDetails()).matches()) {
                    throw new RbacErrorException(error);
                }
                throw error;
            }
        } catch (JsonSyntaxException e) {
            log.debug("Non parsable response error");
        }
    }

    public String traceHttpMessage(HttpMessage message, ProtocolType protocolType) {
        log.info(message.toString());
        Map<String, String> root = new HashMap<>();
        root.put("usualFormatString", message.toString());

        String curlString;
        switch (protocolType) {
            case JSON:
                curlString = message.toCurlStringForJson();
                break;
            case SOAP:
                curlString = message.toCurlStringForSoap();
                break;
            case XML:
                curlString = message.toCurlStringForXml();
                break;
            default:
                curlString = "Curl for this protocol is not implemented";
                break;
        }
        root.put("curlString", curlString);

        root.put("buttonName", protocolType.toString());

        String attachmentHTML;
        try {
            TemplateLoader file = new ClassTemplateLoader(this.getClass(), "/");
            Configuration cfg = new Configuration(VERSION_2_3_0);
            cfg.setTemplateLoader(file);
            cfg.setObjectWrapper(new DefaultObjectWrapper(VERSION_2_3_0));
            cfg.setDefaultEncoding("UTF-8");
            Template template = cfg.getTemplate("request2curlTemplate.ftl");
            Writer fileWriter = new StringWriter();
            try {
                template.process(root, fileWriter);
            } finally {
                fileWriter.close();
            }
            attachmentHTML = fileWriter.toString();
        } catch (Exception e) {
            throw new RuntimeException("Error", e);
        }

        return attachmentHTML;
    }
}
