package ru.yandex.autotests.directapi.apiclient;

import java.nio.charset.Charset;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.gson.JsonSyntaxException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.json.JSONObject;

import ru.yandex.autotests.directapi.apiclient.config.ConnectionConfig;
import ru.yandex.autotests.directapi.apiclient.config.ProtocolType;
import ru.yandex.autotests.directapi.apiclient.errors.Api5JsonError;
import ru.yandex.autotests.directapi.apiclient.errors.RbacErrorException;
import ru.yandex.autotests.directapi.apiclient.internal.JSONRequest;
import ru.yandex.autotests.directapi.apiclient.internal.JsonBuilder;
import ru.yandex.autotests.directapi.apiclient.internal.JsonClientProxyConfig;
import ru.yandex.autotests.directapi.apiclient.internal.MethodInvocationResult;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;
import ru.yandex.autotests.directapi.model.api5.Action;
import ru.yandex.autotests.directapi.model.api5.ServiceNames;
import ru.yandex.qatools.allure.annotations.Attachment;

public class JsonClient extends ApiClient {
    public static Log log = LogFactory.getLog(JsonClient.class);

    private final JsonClientProxyConfig proxyConfig = new JsonClientProxyConfig();

    @Override
    public <T> MethodInvocationResult<T> invokeMethodEx(ServiceNames serviceName, String login, Action methodName, Object params) {
        Class resultType = getReturnClass(serviceName, methodName);
        MethodInvocationResult<String> response = doInvokeMethod(serviceName, login, methodName, params);
        @SuppressWarnings("unchecked")
        T responseObject = (T) JsonBuilder.gsonV5.fromJson(response.getResponseObject(), resultType);
        return new MethodInvocationResult<>(responseObject, response.isJavaResponse());
    }

    public JsonClient(ConnectionConfig connectionConfig, RequestHeader requestHeader) {
        super(connectionConfig, requestHeader);
    }

    @Override
    protected ProtocolType getProtocolType() {
        return ProtocolType.JSON;
    }

    public String sendRequest(ServiceNames serviceName, String login, Action methodName, Object params) {
        return doInvokeMethod(serviceName, login, methodName, params).getResponseObject();
    }

    private MethodInvocationResult<String> doInvokeMethod(ServiceNames serviceName, String login, Action methodName, Object params) {
        JSONRequest request = new JSONRequest();
        request.setHeader(getHttpHeaders(login));
        request.setMethod(methodName.toString());
        request.setParameters(params);

        String endpoint = connectionConfig.getEndPoint(getProtocolType(), serviceName.toString());
        endpoint = proxyConfig.getEndpoint(serviceName, methodName, endpoint);

        //custom object for logging
        HttpMessage message = new HttpMessage();
        message.setUrl(endpoint);
        message.setHeaderMap(request.getHeader());
        message.setRequest(request.toString());

        try {
            CloseableHttpClient httpClient = HttpClientFactory.getHttpClient();
            HttpPost httpRequest = new HttpPost(endpoint);
            StringEntity entity = new StringEntity(request.toString(),
                    ContentType.create("application/json", Charset.forName("UTF-8")));
            httpRequest.setEntity(entity);
            httpRequest.setHeaders(toHeadersArray(request.getHeader()));

            MethodInvocationResult<String> methodInvocationResult = httpClient.execute(httpRequest, new ApiResponseHandler(message));

            String response = methodInvocationResult.getResponseObject();

            if (response == null) {
                throw new DirectAPIException("Получен пустой ответ от сервера");
            }

            try {
                message.setResponse(new JSONObject(response).toString(2));
            } catch (JsonSyntaxException e) {
                throw new DirectAPIException("Получен некорректный ответ от сервера", e);
            }

            traceHttpMessage(message);
            response = getResponseResult(response);
            checkJsonFaultPresence(response, methodInvocationResult.isJavaResponse());

            return new MethodInvocationResult<>(response, methodInvocationResult.isJavaResponse());
        } catch (Api5JsonError | RbacErrorException e) {
            throw e;
        } catch (DirectAPIException knownException) {
            traceHttpMessage(message);
            throw new DirectAPIException("Неверный ответ сервера: " + knownException.getMessage());
        } catch (Exception e) {
            traceHttpMessage(message);
            throw new DirectAPIException("Неизвестная ошибка при выполнении JSON запроса", e);
        }
    }

    @Attachment(value = "[JSON/CURL]: Request", type = "text/html")
    public String traceHttpMessage(HttpMessage message) {
        return super.traceHttpMessage(message, ProtocolType.JSON);
    }

    private Class getReturnClass(ServiceNames serviceName, Action methodName) {
        java.lang.reflect.Method method = null;
        java.lang.reflect.Method[] methods = serviceName.getMethodsClass().getDeclaredMethods();
        for (java.lang.reflect.Method m : methods) {
            if (m.getName().toLowerCase().equals(methodName.toString().toLowerCase())) {
                method = m;
            }
        }
        if (method == null) {
            return String.class;
        } else {
            return method.getReturnType();
        }
    }

    /**
     * Возвращает содержимое структуры result {} или error {}, полученное в ответе сервиса
     *
     * @param response ответ сервиса
     * @return содержимое структуры result{} или error{}, полученное в ответе сервиса в формате json
     */
    public String getResponseResult(String response) {
        String cleanResponse = response.replace(":null", ":\"null\"");
        Pattern p = Pattern.compile("(result|error).*?:(.*)}");
        Matcher m = p.matcher(cleanResponse);
        if (m.find()) {
            return (m.group(2));
        }
        throw new DirectAPIException("Не удалось извлечь result или error из ответа:\n" + response);
    }
}
