package ru.yandex.autotests.directapi.apiclient.config;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;

import javax.xml.ws.WebServiceException;

import com.google.common.collect.ImmutableMap;
import org.apache.http.client.utils.URIBuilder;

import ru.yandex.autotests.directapi.exceptions.DirectAPIException;

import static java.util.Collections.singletonMap;

public class ConnectionConfig implements Cloneable {
    /**
     * api path definitions for each api version and protocol type
     */
    public static final Map<ProtocolType, Map<Integer, String>> API_PATH = ImmutableMap.of(
            ProtocolType.SOAP, singletonMap(5, "/v5/soap/"),
            ProtocolType.JSON, singletonMap(5, "/v5/json/"));
    /**
     * Host like http://test-direct.yandex.ru
     */
    private String apiHost;
    /**
     * api version: JSON | SOAP
     */
    private ProtocolType protocolType;
    /**
     * api version: 4 | 104 | 5
     */
    private int apiVersion = 104;

    public ConnectionConfig(String apiHost) {
        this.apiHost = apiHost;
        this.apiVersion = 5;
        this.protocolType = ProtocolType.JSON;
    }

    public ConnectionConfig clone() {
        return new ConnectionConfig(apiHost)
                .version(apiVersion)
                .protocolType(protocolType);
    }

    public ConnectionConfig version(int apiVersion) {
        this.apiVersion = apiVersion;
        return this;
    }

    public ConnectionConfig protocolType(ProtocolType protocolType) {
        this.protocolType = protocolType;
        return this;
    }

    public int getAPIVersion() {
        return apiVersion;
    }

    public String getApiHost() {
        return apiHost;
    }

    public String getEndPoint(ProtocolType type, String serviceName) {
        try {
            URIBuilder uriBuilder = new URIBuilder(apiHost);

            uriBuilder.setScheme("http");

            if (uriBuilder.getPort() == -1) {
                uriBuilder.setPort(14443);
            }

            switch (type) {
                case SOAP:
                    uriBuilder.setPath("/v5/" + serviceName);
                    break;
                case JSON:
                    uriBuilder.setPath("/json/v5/" + serviceName);
                    break;
                case XML:
                    uriBuilder.setPath("/v5/" + serviceName);
                    break;
            }
            return uriBuilder.build().toString();
        } catch (URISyntaxException e) {
            throw new DirectAPIException("Ошибка формирования пути для запроса API в версии 5", e);
        }
    }

    public URL getV5WsdlLocation(String serviceName) {
        URL url = null;
        try {
            url = new URL(getEndPoint(ProtocolType.SOAP, serviceName) + "?wsdl");
        } catch (MalformedURLException ex) {
            throw new WebServiceException(ex);
        }
        return url;
    }

    public void url(String apiHost) {
        this.apiHost = apiHost;
    }

    public ProtocolType getProtocolType() {
        return protocolType;
    }
}
