package ru.yandex.autotests.directapi.apiclient.errors;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.yandex.direct.api.v5.general.ApiExceptionMessage;

import ru.yandex.autotests.direct.utils.model.ApiException;
import ru.yandex.autotests.direct.utils.textresource.ITextResource;
import ru.yandex.autotests.direct.utils.textresource.TextResourceFormatter;
import ru.yandex.autotests.directapi.apiclient.config.ApiLocale;
import ru.yandex.autotests.directapi.apiclient.internal.JsonUtils;
import ru.yandex.autotests.directapi.exceptions.DirectAPIException;

/**
 * Author pavryabov
 * Date 17.07.14
 */
public class Api5Error extends RuntimeException implements Api5ErrorInfo {
    private String message;
    private ApiExceptionMessage faultInfo;
    private Object doEqualsCalc = null;
    private boolean doHashCodeCalc = false;
    // Для того чтобы исключить данное поле из сериализации в строку
    private transient boolean javaResponse;

    public Api5Error(ApiException apiException) {
        this(apiException, false);
    }

    public Api5Error(ApiException apiException, boolean isJavaResponse) {
        this.message = apiException.getMessage();
        this.faultInfo = new ApiExceptionMessage();
        this.faultInfo.setErrorCode(apiException.getFaultInfo().getErrorCode());
        this.faultInfo.setErrorDetail(apiException.getFaultInfo().getErrorDetail());
        this.javaResponse = isJavaResponse;
    }

    public Api5Error(int code) {
        this(code, Api5ErrorDetails.EMPTY_STRING);
    }

    public Api5Error(int code, Api5ErrorDetails details) {
        this(code, details, new Object[0]);
    }

    public Api5Error(int code, Api5ErrorDetails details, Object... detailsParams) {
        init(code, Api5ErrorMessage.ERROR_MESSAGES, details, detailsParams);
    }

    public Api5Error(int code, Api5ErrorDetailsJava details, Object... detailsParams) {
        init(code, Api5ErrorMessage.ERROR_MESSAGES, details, detailsParams);
    }

    public static String enumAsParam(Class<? extends Enum> enumClass) {
        Method method = null;
        Method[] declaredMethods = enumClass.getDeclaredMethods();
        for (Method m : declaredMethods) {
            if ("value".equals(m.getName())) {
                method = m;
                break;
            }
            if ("name".equals(m.getName())) {
                method = m;
            }
        }
        final Method m = method;
        return Stream.of(enumClass.getEnumConstants())
                .map(e -> {
                    try {
                        return (String) (m != null ? m.invoke(e) : null);
                    } catch (IllegalAccessException | InvocationTargetException e1) {
                        // ignore
                    }
                    return null;
                })
                .collect(Collectors.joining(", "));
    }

    public String getMessage() {
        return message;
    }

    public void setMessage(String message) {
        this.message = message;
    }

    @SuppressWarnings("WeakerAccess")
    public void setMessageFromResource(Api5ErrorMessage message) {
        TextResourceFormatter messageFormatter = TextResourceFormatter
                .resource(message)
                .locale(ApiLocale.RU.toString());
        this.message = messageFormatter.toString();
    }

    public Api5Error withMessageFromResource(Api5ErrorMessage message) {
        setMessageFromResource(message);
        return this;
    }

    public ApiExceptionMessage getFaultInfo() {
        return faultInfo;
    }

    public void setFaultInfo(ApiExceptionMessage faultInfo) {
        this.faultInfo = faultInfo;
    }

    private void init(int code,
                      Map<Integer, Api5ErrorMessage> messagesMap,
                      ITextResource details,
                      Object... detailsParams) {
        Api5ErrorMessage message = messagesMap.get(code);
        if (message == null) {
            throw new DirectAPIException("Отсутствует маппинг кода ошибки и соответстующего сообщения. Код - " + code);
        }
        setMessageFromResource(message);
        TextResourceFormatter detailsFormatter = TextResourceFormatter
                .resource(details)
                .args(detailsParams)
                .locale(ApiLocale.RU.toString());
        this.faultInfo = new ApiExceptionMessage();
        this.faultInfo.setErrorCode(code);
        this.faultInfo.setErrorDetail(detailsFormatter.toString());
    }

    public Api5JsonError toJsonError() {
        return new Api5JsonError()
                .setDetails(this.faultInfo.getErrorDetail())
                .setErrorCode(this.faultInfo.getErrorCode())
                .setMessage(message);
    }

    public Api5XmlError toXmlError() {
        Api5XmlError xmlError = new Api5XmlError()
                .setErrorDetail(this.faultInfo.getErrorDetail())
                .setErrorCode(this.faultInfo.getErrorCode())
                .setErrorMessage(message);
        return xmlError;
    }

    public boolean isJavaResponse() {
        return javaResponse;
    }

    public void setJavaResponse(boolean javaResponse) {
        this.javaResponse = javaResponse;
    }

    public String toString() {
        return JsonUtils.toString(this);
    }

    public String toExpandString() {
        StringBuilder sb = new StringBuilder(JsonUtils.toString(this));
        return sb.toString();
    }

    @Override
    public synchronized boolean equals(Object got) {
        if (!(got instanceof Api5Error)) {
            return false;
        }
        Api5Error other = (Api5Error) got;
        if (got == null) {
            return false;
        }
        if (this == got) {
            return true;
        }
        if (doEqualsCalc != null) {
            return (doEqualsCalc == got);
        }
        doEqualsCalc = got;
        try {
            return (this.message == other.getMessage()) && this.faultInfo.equals(other.getFaultInfo());
        } finally {
            doEqualsCalc = null;
        }
    }

    @Override
    public Integer getErrorCode() {
        return faultInfo.getErrorCode();
    }

    @Override
    public String getErrorMessage() {
        return message;
    }

    @Override
    public String getErrorDetails() {
        return faultInfo.getErrorDetail();
    }

    public synchronized int hashCode() {
        if (doHashCodeCalc) {
            return 0;
        }
        doHashCodeCalc = true;
        int hash = 1;
        if (getMessage() != null) {
            hash += getMessage().hashCode();
        }
        if (getFaultInfo() != null) {
            hash += getFaultInfo().hashCode();
        }
        doHashCodeCalc = false;
        return hash;
    }
}
