package ru.yandex.autotests.directapi.apiclient.errors;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableMap;

import ru.yandex.autotests.direct.utils.textresource.ITextResource;
import ru.yandex.direct.core.validation.CommonDefectTranslations;
import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.bundle.TranslationBundle;
import ru.yandex.direct.i18n.types.DummyTranslatable;

/**
 * Author pavryabov
 * Date 18.07.14
 * Список ключей к описаниям ошибок API5.
 * Тексты ошибок содержатся в ресурсах локализации
 *
 * @see /APIAxisErrorMessage
 */
public enum Api5ErrorMessage implements ITextResource {
    AD_HAS_ALREADY_ARCHIVED,
    AD_HAS_ALREADY_RESUMED,
    AD_HAS_ALREADY_SUSPENDED,
    AD_NOT_ARCHIVED,
    ARCHIEVED_OBJECT_EDIT_NOT_ALLOWED,
    AUTHORIZATION_ERROR,
    AUTOTARGETING_ALREADY_SUSPENDED,
    AUTOTARGETING_HAS_NOT_BEEN_STOPPED,
    BAD_STATUS(CommonDefectTranslations.class, CommonDefectTranslations::badStatusShort),
    BID_CAN_NOT_CHANGED,
    BID_WILL_NOT_APPLIED,
    CANT_CREATE_LOGIN,
    CANT_PROCESS_REPORT_IN_ONLINE_MODE,
    COULD_NOT_ARCHIVE_OBJECT,
    COULD_NOT_UNARCHIVE_OBJECT,
    DAILY_BUDGET_IS_RESET,
    ELEMENT_CANT_PRESENT_IN_LIST_MORE_THAN_ONE_TIME(CommonDefectTranslations.class,
            CommonDefectTranslations::duplicatedArrayElementShort),
    EXCEEDED_OBJECTS_LIMIT_FOR_OPERATION,
    EXCEEDED_OBJECTS_QUEUE,
    EXCEED_LIMIT_CONNECTIONS,
    EXCEED_LIMIT_OF_REQUESTS,
    FIELD_NOT_MATCH_STRATEGY,
    FIELD_NOT_PASSED,
    FIELD_SET_INCORRECTLY(CommonDefectTranslations.class, CommonDefectTranslations::invalidValueShort),
    GROUP_CONTAIN_OBJECTS,
    GROUP_NAME_CANNOT_BE_EMPTY,
    IMPRESSIONS_ARE_RESTRICTED_IN_SEVERAL_REGIONS,
    INCOMPATIBLE_OPERATIONS_IN_ONE_REQUEST,
    INCOMPLETE_REGISTRATION,
    INCONSISTENT_OBJECT_STATE(CommonDefectTranslations.class, CommonDefectTranslations::inconsistentStateShort),
    INCORRECT_USE_OF_BRACKETS_IN_MINUS_PHRASE(CommonDefectTranslations.class,
            CommonDefectTranslations::invalidCharsShort),
    INCORRECT_USE_OF_NEGATIVE_KEYWORDS(CommonDefectTranslations.class, CommonDefectTranslations::minusWordsShort),
    INCORRECT_USE_OF_STOPWORD(CommonDefectTranslations.class, CommonDefectTranslations::stopWordsShort),
    INVALID_ACTION,
    INVALID_FORMAT(CommonDefectTranslations.class, CommonDefectTranslations::invalidFormatShort),
    INVALID_GEOTARGETING,
    INVALID_GEO_DATA_TO_DISPLAY_A_MAP,
    INVALID_OBJECT_STATUS(CommonDefectTranslations.class, CommonDefectTranslations::badStatusShort),
    INVALID_OFFSET,
    INVALID_PARAMETERS_OF_PAGE,
    INVALID_REQUEST,
    INVALID_REQUEST_FORMAT,
    INVALID_REQUEST_PARAMETERS(CommonDefectTranslations.class, CommonDefectTranslations::invalidRequestParamsShort),
    INVALID_SELECTION_CRITERIA,
    INVALID_TRACKING_URL,
    INVALID_USE_OF_FIELD,
    INVALID_VALUE(CommonDefectTranslations.class, CommonDefectTranslations::invalidValueShort),
    IN_ONE_REQUEST_SHOULD_BE_ONE_TYPE_ID,
    KEYWORD_ALREADY_SUSPENDED,
    KEYWORD_CONTAINS_TOO_MANY_WORDS(CommonDefectTranslations.class, CommonDefectTranslations::maxWordsShort),
    KEYWORD_DUPLICATES,
    KEYWORD_HAS_NOT_BEEN_STOPPED,
    KEYWORD_LENGTH_EXCEEDED(CommonDefectTranslations.class, CommonDefectTranslations::keywordMaxLengthShort),
    KEYWORD_MAX_LENGTH(CommonDefectTranslations.class, CommonDefectTranslations::keywordMaxLengthShort),
    KEYWORD_MINUS_WORD_LENGTH_EXCEEDED,
    KEYWORD_NOT_FOUND,
    KEYWORD_WAS_CREATED_USING_UPDATE,
    LANGUAGE_DOESNT_MATCH_GEOTARGETING,
    LIMIT_ACCESS,
    LOGIN_IS_IN_USE,
    LOGIN_IS_OCCUPIED,
    LOGIN_NOT_SWITCHED_ON_TO_DIRECT,
    MANDATORY_FIELD(CommonDefectTranslations.class, CommonDefectTranslations::requiredButEmptyShort),
    MAXIMUM_LENGTH_EXCEEDED(CommonDefectTranslations.class, CommonDefectTranslations::maxStringSizeShort),
    MAXIMUM_NUMBER_OF_OBJECTS_REACHED(CommonDefectTranslations.class,
            CommonDefectTranslations::maxElementsExceededShort),
    MAX_FILE_SIZE_EXCEEDED,
    MAX_WORDS(CommonDefectTranslations.class, CommonDefectTranslations::maxWordsShort),
    MINUS_WORDS(CommonDefectTranslations.class, CommonDefectTranslations::minusWordsShort),
    MORE_THEN_ONE_FIELD_PASSED,
    NAME_LENGTH_CANNOT_BE_MORE_THAN_255,
    NEGATIVE_KEYWORD_CANNOT_BE_PHRASE,
    NOT_ENOUGH_RIGHTS(CommonDefectTranslations.class, CommonDefectTranslations::noRightsShort),
    NOT_ENOUGH_UNITS,
    NOT_POSSIBLE_TO_DELETE_OBJECT,
    NOT_POSSIBLE_TO_PAUSE_IMPRESSIONS,
    NOT_SUPPORTED,
    NO_ACCESS_TO_API,
    NO_ACCESS_TO_METHOD,
    NO_EFFECT,
    NO_ONE_OF_REQUIRED_FIELDS,
    NO_ONE_OF_REQUIRED_PARAMETERS,
    NUMBER_OF_PERMITED_ELEMENTS_EXCEEDED_OR_EQUALS_ZERO,
    OBJECTS_ALREADY_LINKED,
    OBJECTS_NOT_LINKED,
    OBJECT_ALREADY_ARCHIVED,
    OBJECT_ALREADY_SUSPENDED,
    OBJECT_CANNOT_BE_MORE_THAN_ONCE_IN_REQUEST,
    OBJECT_IS_MARKED_AS_DELETED,
    OBJECT_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST,
    OBJECT_NOT_ARCHIVED,
    OBJECT_NOT_FOUND(CommonDefectTranslations.class, CommonDefectTranslations::notFoundShort),
    OBJECT_NOT_SUSPENDED,
    OBJECT_WITH_SUCH_PARAMS_ALREADY_EXISTS,
    OPERATION_ERROR,
    OPERATION_NOT_FOUND,
    PARAM_WILL_NOT_BE_CHANGED,
    PRIORITY_WILL_NOT_CHANGED,
    PUT_MORE_THAN_ONE_PARAMETER,
    REGION_DOESNT_MATCH_ADS_TEXT,
    RELEVANT_KEYWORDS_DISABLED,
    REPEATING_ELEMENT,
    REQUIRED_BUT_EMPTY(CommonDefectTranslations.class, CommonDefectTranslations::requiredButEmptyShort),
    SETTING_WILL_NOT_BE_CHANGED(CommonDefectTranslations.class, CommonDefectTranslations::settingWillNotBeChanged),
    SET_EMPTY_VALUE,
    SITELINKS_DUPLICATES,
    SITELINKS_SET_USED_AND_CANT_BE_DELETED,
    STOP_WORDS(CommonDefectTranslations.class, CommonDefectTranslations::stopWordsShort),
    TOO_WIDE_SELECTION_CRITERIA,
    TRACKING_URL_DOESNT_CONTAIN_LOGID,
    UNABLE_DELETE_OBJECT,
    UNSUPPORTED_APPLICATION_STORE,
    USES_INADMISSIBLE_CHARACTERS(CommonDefectTranslations.class, CommonDefectTranslations::invalidCharsShort),
    VCARD_DUPLICATES,
    WEBPAGE_NOT_SUSPENDED,
    WEBPAGE_SUSPENDED,
    WRONG_OBJECT_TYPE,
    ;

    public static final Map<Integer, Api5ErrorMessage> ERROR_MESSAGES =
            ImmutableMap.<Integer, Api5ErrorMessage>builder()
                    .put(1005, OPERATION_ERROR)
                    .put(1016, INVALID_OFFSET)
                    .put(152, NOT_ENOUGH_UNITS)
                    .put(252, LOGIN_IS_OCCUPIED)
                    .put(253, CANT_CREATE_LOGIN)
                    .put(3000, NO_ACCESS_TO_API)
                    .put(3001, NO_ACCESS_TO_METHOD)
                    .put(4000, INVALID_REQUEST_PARAMETERS)
                    .put(4001, INVALID_SELECTION_CRITERIA)
                    .put(4002, INVALID_PARAMETERS_OF_PAGE)
                    .put(4003, NO_ONE_OF_REQUIRED_PARAMETERS)
                    .put(4004, PUT_MORE_THAN_ONE_PARAMETER)
                    .put(4005, IN_ONE_REQUEST_SHOULD_BE_ONE_TYPE_ID)
                    .put(5002, USES_INADMISSIBLE_CHARACTERS)
                    .put(5005, FIELD_SET_INCORRECTLY)
                    .put(5008, NO_ONE_OF_REQUIRED_FIELDS)
                    .put(506, EXCEED_LIMIT_CONNECTIONS)
                    .put(513, LOGIN_NOT_SWITCHED_ON_TO_DIRECT)
                    .put(5140, KEYWORD_CONTAINS_TOO_MANY_WORDS)
                    .put(5141, INCORRECT_USE_OF_STOPWORD)
                    .put(5142, KEYWORD_LENGTH_EXCEEDED)
                    .put(5161, INCORRECT_USE_OF_NEGATIVE_KEYWORDS)
                    .put(5200, LOGIN_IS_IN_USE)
                    .put(53, AUTHORIZATION_ERROR)
                    .put(54, NOT_ENOUGH_RIGHTS)
                    .put(55, OPERATION_NOT_FOUND)
                    .put(56, EXCEED_LIMIT_OF_REQUESTS)
                    .put(58, INCOMPLETE_REGISTRATION)
                    .put(6000, INCONSISTENT_OBJECT_STATE)
                    .put(7000, NUMBER_OF_PERMITED_ELEMENTS_EXCEEDED_OR_EQUALS_ZERO)
                    .put(7001, MAXIMUM_NUMBER_OF_OBJECTS_REACHED)
                    .put(8000, INVALID_REQUEST_FORMAT)
                    .put(8312, CANT_PROCESS_REPORT_IN_ONLINE_MODE)
                    .put(8800, OBJECT_NOT_FOUND)
                    .put(9000, EXCEEDED_OBJECTS_QUEUE)
                    .put(9300, EXCEEDED_OBJECTS_LIMIT_FOR_OPERATION)
                    .put(9301, TOO_WIDE_SELECTION_CRITERIA)
                    .put(9800, OBJECT_CANNOT_BE_MORE_THAN_ONCE_IN_REQUEST)
                    .put(9802, ELEMENT_CANT_PRESENT_IN_LIST_MORE_THAN_ONE_TIME)
                    .build();
    public static final Map<Integer, Api5ErrorMessage> NOTIFICATION_MESSAGES = ImmutableMap.<Integer,
                    Api5ErrorMessage>builder()
            .put(54, NOT_ENOUGH_RIGHTS)
            .put(1002, AUTHORIZATION_ERROR)
            .put(1003, OPERATION_NOT_FOUND)
            .put(1012, MORE_THEN_ONE_FIELD_PASSED)
            .put(1013, FIELD_NOT_PASSED)
            .put(1014, ARCHIEVED_OBJECT_EDIT_NOT_ALLOWED)
            .put(3500, NOT_SUPPORTED)
            .put(3600, LIMIT_ACCESS)
            .put(4000, INVALID_REQUEST_PARAMETERS)
            .put(4003, NO_ONE_OF_REQUIRED_PARAMETERS)
            .put(4004, PUT_MORE_THAN_ONE_PARAMETER)
            .put(4006, INCOMPATIBLE_OPERATIONS_IN_ONE_REQUEST)
            .put(5000, MANDATORY_FIELD)
            .put(5001, MAXIMUM_LENGTH_EXCEEDED)
            .put(5002, USES_INADMISSIBLE_CHARACTERS)
            .put(5003, SET_EMPTY_VALUE)
            .put(5004, INVALID_FORMAT)
            .put(5005, FIELD_SET_INCORRECTLY)
            .put(5006, INVALID_USE_OF_FIELD)
            .put(5007, REPEATING_ELEMENT)
            .put(5008, NO_ONE_OF_REQUIRED_FIELDS)
            .put(5009, MORE_THEN_ONE_FIELD_PASSED)
            .put(5010, MAX_FILE_SIZE_EXCEEDED)
            .put(5120, INVALID_GEOTARGETING)
            .put(5140, KEYWORD_CONTAINS_TOO_MANY_WORDS)
            .put(5141, INCORRECT_USE_OF_STOPWORD)
            .put(5142, KEYWORD_LENGTH_EXCEEDED)
            .put(5160, NEGATIVE_KEYWORD_CANNOT_BE_PHRASE)
            .put(5161, INCORRECT_USE_OF_NEGATIVE_KEYWORDS)
            .put(5162, KEYWORD_MINUS_WORD_LENGTH_EXCEEDED)
            .put(5180, UNSUPPORTED_APPLICATION_STORE)
            .put(6000, INCONSISTENT_OBJECT_STATE)
            .put(6001, WRONG_OBJECT_TYPE)
            .put(6100, REGION_DOESNT_MATCH_ADS_TEXT)
            .put(6101, LANGUAGE_DOESNT_MATCH_GEOTARGETING)
            .put(7000, NUMBER_OF_PERMITED_ELEMENTS_EXCEEDED_OR_EQUALS_ZERO)
            .put(7001, MAXIMUM_NUMBER_OF_OBJECTS_REACHED)
            .put(7002, INVALID_TRACKING_URL)
            .put(8300, INVALID_OBJECT_STATUS)
            .put(8301, UNABLE_DELETE_OBJECT)
            .put(8302, NOT_POSSIBLE_TO_PAUSE_IMPRESSIONS)
            .put(8303, COULD_NOT_ARCHIVE_OBJECT)
            .put(8304, COULD_NOT_UNARCHIVE_OBJECT)
            .put(8305, INVALID_ACTION)
            .put(8000, INVALID_REQUEST)
            .put(8800, OBJECT_NOT_FOUND)
            .put(9600, FIELD_NOT_MATCH_STRATEGY)
            .put(9601, BID_CAN_NOT_CHANGED)
            .put(9800, OBJECT_CANNOT_BE_MORE_THAN_ONCE_IN_REQUEST)
            .put(9801, OBJECT_WITH_SUCH_PARAMS_ALREADY_EXISTS)
            .put(9802, ELEMENT_CANT_PRESENT_IN_LIST_MORE_THAN_ONE_TIME)
            .put(10000, OBJECT_IS_PRESENTED_MORE_THAN_ONCE_IN_REQUEST)
            .put(10120, SITELINKS_DUPLICATES)
            .put(10140, KEYWORD_DUPLICATES)
            .put(10141, KEYWORD_WAS_CREATED_USING_UPDATE)
            .put(10160, BID_WILL_NOT_APPLIED)
            .put(10161, PRIORITY_WILL_NOT_CHANGED)
            .put(10162, DAILY_BUDGET_IS_RESET)
            .put(10164, OBJECT_WITH_SUCH_PARAMS_ALREADY_EXISTS)
            .put(10172, IMPRESSIONS_ARE_RESTRICTED_IN_SEVERAL_REGIONS)
            .put(10200, AD_HAS_ALREADY_SUSPENDED)
            .put(10201, AD_HAS_ALREADY_RESUMED)
            .put(10202, AD_HAS_ALREADY_ARCHIVED)
            .put(10203, AD_NOT_ARCHIVED)
            .put(10240, KEYWORD_ALREADY_SUSPENDED)
            .put(10241, KEYWORD_HAS_NOT_BEEN_STOPPED)
            .put(10244, AUTOTARGETING_ALREADY_SUSPENDED)
            .put(10245, AUTOTARGETING_HAS_NOT_BEEN_STOPPED)
            .put(102002, GROUP_NAME_CANNOT_BE_EMPTY)
            .put(102003, NAME_LENGTH_CANNOT_BE_MORE_THAN_255)
            .put(102007, GROUP_CONTAIN_OBJECTS)
            .put(10100, VCARD_DUPLICATES)
            .put(10020, OBJECT_ALREADY_SUSPENDED)
            .put(10021, OBJECT_NOT_SUSPENDED)
            .put(10022, OBJECT_ALREADY_ARCHIVED)
            .put(10023, OBJECT_NOT_ARCHIVED)
            .put(10025, OBJECT_IS_MARKED_AS_DELETED)
            .put(10030, NO_EFFECT)
            .put(10035, INVALID_USE_OF_FIELD)
            .put(10163, SETTING_WILL_NOT_BE_CHANGED)
            .put(10165, PARAM_WILL_NOT_BE_CHANGED)
            .put(10170, OBJECTS_ALREADY_LINKED)
            .put(10171, OBJECTS_NOT_LINKED)
            .put(10173, RELEVANT_KEYWORDS_DISABLED)
            .put(10242, WEBPAGE_SUSPENDED)
            .put(10243, WEBPAGE_NOT_SUSPENDED)
            .put(10250, TRACKING_URL_DOESNT_CONTAIN_LOGID)
            .put(107002, SITELINKS_SET_USED_AND_CANT_BE_DELETED)
            .put(171004, KEYWORD_NOT_FOUND)
            .put(209001, INVALID_GEO_DATA_TO_DISPLAY_A_MAP)
            .put(1000115, NOT_POSSIBLE_TO_DELETE_OBJECT)
            .build();
    private final Supplier<Translatable> translatableSupplier;

    <T extends TranslationBundle> Api5ErrorMessage() {
        translatableSupplier = () -> new DummyTranslatable("This message is not translated yet");
    }

    <T extends TranslationBundle> Api5ErrorMessage(Class<T> bundleClass,
                                                   Function<T, Translatable> textProducer) {
        T bundle = bundleClass.cast(I18NBundle.implement(bundleClass));
        translatableSupplier = () -> textProducer.apply(bundle);
    }

    public static Api5ErrorMessage fromNotificationCode(int notificationCode) {
        return NOTIFICATION_MESSAGES.get(notificationCode);
    }

    public static Api5ErrorMessage fromErrorCode(int errorCode) {
        return ERROR_MESSAGES.get(errorCode);
    }

    public Translatable getTranslatable() {
        return translatableSupplier.get();
    }

    @Override
    public String getBundle() {
        throw new UnsupportedOperationException();
    }

    @Override
    public Translatable getTranslatable(List<Object> ignoredParams) {
        return translatableSupplier.get();
    }
}
