package ru.yandex.autotests.directapi.authentication;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nonnull;

import org.mockito.Mockito;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.api.v5.security.ApiAuthenticationManager;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.security.DirectApiPreAuthentication;
import ru.yandex.direct.api.v5.security.SecurityErrors;
import ru.yandex.direct.api.v5.security.internal.DirectApiInternalAuthRequest;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

@Lazy
public class DummyAuthenticationManager implements ApiAuthenticationManager {
    private final ShardHelper shardHelper;
    private ClientService clientService;
    private UserService userService;
    private final ApiAuthenticationSource apiAuthenticationSource;

    public DummyAuthenticationManager(ShardHelper shardHelper,
                                      ClientService clientService, UserService userService,
                                      ApiAuthenticationSource authenticationSource) {
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.userService = userService;
        this.apiAuthenticationSource = authenticationSource;
    }

    @Nonnull
    @Override
    public DirectApiPreAuthentication checkApiAccessAllowed(DirectApiInternalAuthRequest authRequest) {

//      Operator
        String operatorLogin = Optional
                .ofNullable(DummyAuthenticationContainer.getLogin(authRequest.getCredentials().getOauthToken()))
                .orElseThrow(() -> SecurityErrors.newInvalidOauthTokenFormat(new Exception("Invalid token")));
        Long operatorUid = shardHelper.getUidByLogin(operatorLogin);
        User operator = userService.getUser(operatorUid);
        ApiUser apiOperator = convertUserToApiUser(operator);

//      ChiefOperator
        ApiUser apiChiefOperator;
        if (Objects.equals(operatorUid, operator.getChiefUid())) {
            apiChiefOperator = apiOperator;
        } else {
            User chiefOperatopr = userService.getUser(operator.getChiefUid());
            apiChiefOperator = convertUserToApiUser(chiefOperatopr);
        }

//      Subclient
        String subLogin = Optional.ofNullable(authRequest.getCredentials().getClientLogin()).orElse(operatorLogin);
        User subUser = userService.getUserByLogin(subLogin);
        ApiUser apiSubclient = convertUserToApiUser(subUser);

//      ChiefSubclient
        ApiUser apiChiefSubclient;
        if (Objects.equals(subUser.getUid(), subUser.getChiefUid())) {
            apiChiefSubclient = apiSubclient;
        } else {
            User chiefSubclient = userService.getUser(subUser.getChiefUid());
            apiChiefSubclient = convertUserToApiUser(chiefSubclient);
        }

        if (Mockito.mockingDetails(apiAuthenticationSource).isMock()) {
            Mockito.when(apiAuthenticationSource.getOperator()).thenReturn(apiOperator);
            Mockito.when(apiAuthenticationSource.getChiefOperator()).thenReturn(apiChiefOperator);
            Mockito.when(apiAuthenticationSource.getSubclient()).thenReturn(apiSubclient);
            Mockito.when(apiAuthenticationSource.getChiefSubclient()).thenReturn(apiChiefSubclient);
        }

        return new DirectApiPreAuthentication(authRequest.getCredentials(), "Autotests",
                apiOperator, apiChiefOperator, apiSubclient, apiChiefSubclient);
    }

    private static ApiUser convertUserToApiUser(User user) {
        ApiUser apiUser = new ApiUser()
                .withClientId(user.getClientId())
                .withLogin(user.getLogin())
                .withUid(user.getUid())
                .withStatusBlocked(user.getStatusBlocked())
                .withStatusArch(user.getStatusArch())
                .withRole(user.getRole())
                .withEmail(user.getEmail())
                .withFio(user.getFio())
                .withLang(user.getLang())
                .withPhone(user.getPhone())
                .withSendNews(user.getSendNews())
                .withSendAccNews(user.getSendAccNews())
                .withSendWarn(user.getSendWarn());
        // Методы ниже возвращают объект User, а не ApiUser
        apiUser.withRepType(user.getRepType())
                .withGeoId(user.getGeoId())
                .withCanManagePricePackages(user.getCanManagePricePackages())
                .withIsReadonlyRep(user.getIsReadonlyRep())
                .withCanApprovePricePackages(user.getCanApprovePricePackages());
        return apiUser;
    }

    @Nonnull
    @Override
    public DirectApiPreAuthentication checkClientLoginAccess(DirectApiPreAuthentication authenticationCarrier) {
        return authenticationCarrier;
    }
}
