package ru.yandex.autotests.directapi.configuration;

import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

import io.lettuce.core.RedisURI;
import io.lettuce.core.cluster.RedisClusterClient;
import io.lettuce.core.resource.ClientResources;
import one.util.streamex.IntStreamEx;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import redis.embedded.RedisExecProvider;
import redis.embedded.RedisServer;
import redis.embedded.cluster.RedisCluster;
import redis.embedded.util.JedisUtil;
import redis.embedded.util.OS;

import ru.yandex.autotests.directapi.authentication.DummyApiTokenAuthProvider;
import ru.yandex.autotests.directapi.authentication.DummyAuthenticationManager;
import ru.yandex.direct.api.v5.configuration.WebServiceConfiguration;
import ru.yandex.direct.api.v5.security.ApiAuthenticationManager;
import ru.yandex.direct.api.v5.security.ApiAuthenticationSource;
import ru.yandex.direct.api.v5.security.token.ApiTokenAuthProvider;
import ru.yandex.direct.api.v5.testing.configuration.Api5TestingConfiguration;
import ru.yandex.direct.balance.client.BalanceClient;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.jetty.JettyConfig;
import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.DirectConfigFactory;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.testing.stub.BalanceClientStub;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.logging.LoggingInitializer;
import ru.yandex.direct.logging.LoggingInitializerParams;
import ru.yandex.direct.logging.LoggingInitializerUtils;
import ru.yandex.direct.utils.HostPort;
import ru.yandex.direct.utils.NamedThreadFactory;
import ru.yandex.misc.ip.IpPortUtils;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.spy;
import static ru.yandex.direct.api.v5.ApiConstants.DEFAULT_LOCALE;
import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;

@Configuration
@Import({Api5TestingConfiguration.class, WebServiceConfiguration.class})
public class AutotestsConfiguration {
    public static final String API_INSTANCE = "api";

    private static final ExecutorService executorService = Executors.newSingleThreadExecutor();
    private static final Object lock = new Object();
    private static volatile AutotestsConfiguration instance;
    private static volatile JettyInstance apiInstance;

    public static AutotestsConfiguration getInstance() {
        if (instance == null) {
            synchronized (lock) {
                if (instance == null) {
                    instance = new AutotestsConfiguration();
                }
            }
        }
        return instance;
    }

    private static JettyInstance getApiInstance() {
        if (apiInstance == null) {
            synchronized (lock) {
                if (apiInstance == null) {
                    try {
                        apiInstance = startJetty(API_INSTANCE, AutotestsConfiguration.class);
                    } catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
            }
        }
        return apiInstance;
    }

    private static JettyInstance startJetty(String serviceName, Class<?> applicationClass)
            throws InterruptedException {
        Locale.setDefault(DEFAULT_LOCALE);
        LoggingInitializerParams loggingParams = new LoggingInitializerParams();
        LoggingInitializerUtils.allowReinitialization();
        LoggingInitializer.initialize(loggingParams, serviceName);
        DirectConfig directConfig = DirectConfigFactory.getConfig(EnvironmentType.DB_TESTING);
        JettyConfig config = new JettyConfig(directConfig.getBranch("jetty"));
        TestJettyLauncher launcher = TestJettyLauncher.server(config);
        try {
            launcher.getStartupLock().lock();
            executorService.submit(() -> {
                try {
                    launcher
                            .withDefaultWebApp(applicationClass.getClassLoader(), "")
                            .run();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            });
        } finally {
            launcher.getStartupLock().unlock();
        }
        return new JettyInstance(launcher.hostPort(), launcher.getWebAppContexts().get(0));
    }

    @Lazy
    @Bean
    @Qualifier(API_INSTANCE)
    public JettyInstance api() {
        return getApiInstance();
    }

    @Lazy
    @Primary
    @Bean
    public RedisClusterClient lettuce(RedisCluster cluster, ClientResources resources) {
        List<RedisURI> redisURIs = JedisUtil.jedisHosts(cluster).stream()
                .map(HostPort::fromString)
                .map(h -> RedisURI.create(h.getHost(), h.getPort()))
                .collect(Collectors.toList());
        return RedisClusterClient.create(redisURIs);
    }

    @Lazy
    @Primary
    @Bean
    @Qualifier(LETTUCE)
    public LettuceConnectionProvider lettuceConnectionProvider(RedisClusterClient clusterClient) {
        return new LettuceConnectionProvider("test", clusterClient, 3);
    }

    @Bean
    @Primary
    public ApiAuthenticationManager apiAuthenticationManager(ShardHelper shardHelper,
                                                             ClientService clientService,
                                                             UserService userService,
                                                             ApiAuthenticationSource authenticationSource) {
        return new DummyAuthenticationManager(shardHelper, clientService, userService, authenticationSource);
    }

    @Bean
    @Primary
    public PpcPropertiesSupport ppcPropertiesSupport(DslContextProvider dslContextProvider) {
        return new TestPpcPropertiesSupport(dslContextProvider);
    }

    @Bean
    @Primary
    public ApiTokenAuthProvider apiTokenAuthProvider(ShardHelper shardHelper) {
        return new DummyApiTokenAuthProvider(shardHelper);
    }

    @Bean
    @Primary
    ApiAuthenticationSource apiAuthenticationSource(DirectConfig cfg) {
        return new ApiAuthenticationSource(
                cfg.getStringList("services_application_ids"),
                cfg.getStringList("display_url_texts_allowed_application_ids"),
                cfg.getStringList("leadform_attributes_allowed_application_ids"),
                Collections.emptyMap()
        );
    }

    //MockBean не работают, когда конфигурация запускается кодом, поэтому оставляем spy
    @Bean
    public BalanceClient balanceClient(DatabaseWrapperProvider databaseWrapperProvider) {
        return spy(new BalanceClientStub(databaseWrapperProvider));
    }

    //MockBean не работают, когда конфигурация запускается кодом, поэтому оставляем mock
    @Bean
    public BlackboxUserService blackboxUserService() {
        return mock(BlackboxUserService.class);
    }

    @Bean
    public RedisCluster redisCluster() {
        String pathToRedis = RedisDownloader.getRedisPath();

        RedisExecProvider provider = RedisExecProvider.build()
                .override(OS.UNIX, pathToRedis)
                .override(OS.MAC_OS_X, pathToRedis);

        RedisCluster cluster = new RedisCluster.Builder()
                .numOfMasters(3)
                .numOfRetries(10)
                .serverPorts(IntStreamEx
                        .range(3)
                        .mapToObj(i -> IpPortUtils.getFreeLocalPort(10000, 20000).getPort())
                        .toList())
                .withServerBuilder(new RedisServer.Builder().redisExecProvider(provider))
                .build();
        try {
            cluster.start();
        } catch (Exception e) {
            throw new RuntimeException("cannot start redis", e);
        } finally {
            Thread shutdownThread = new NamedThreadFactory("RedisShutdownHook").newThread(cluster::stop);
            Runtime.getRuntime().addShutdownHook(shutdownThread);
        }
        return cluster;
    }

}
