package ru.yandex.autotests.directapi.configuration;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.NamedThreadFactory;
import ru.yandex.direct.utils.io.FileUtils;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static ru.yandex.devtools.test.Paths.getSandboxResourcesRoot;

class RedisDownloader {
    private static final Logger logger = LoggerFactory.getLogger(RedisDownloader.class);
    private static final String REDIS_SERVER_BINARY_NAME = "redis-server";

    private static final String TARBALL_FILENAME_LINUX = "redis-3.2.6.tar.gz";
    private static final String TARBALL_SANDBOX_URL_LINUX = "https://proxy.sandbox.yandex-team.ru/204417534";

    private static final String TARBALL_FILENAME_MAC_OS_X = "redis-3.2.6_mac_os_x.tar.gz";
    private static final String TARBALL_SANDBOX_URL_MAC_OS_X = "https://proxy.sandbox.yandex-team.ru/1182831412";

    static String getRedisPath() {
        String osName = System.getProperty("os.name");

        switch (osName) {
            case "Linux":
                return getRedisPath(TARBALL_FILENAME_LINUX, TARBALL_SANDBOX_URL_LINUX);
            case "Mac OS X":
                return getRedisPath(TARBALL_FILENAME_MAC_OS_X, TARBALL_SANDBOX_URL_MAC_OS_X);
        }
        throw new IllegalArgumentException();
    }

    private static String getRedisPath(String tarballFilename, String tarballSandboxUrl) {
        Path tarballPath = Paths.get(tarballFilename);
        boolean isSandboxEnv = getSandboxResourcesRoot() != null;

        try {
            if (!tarballPath.toFile().exists()) {
                if (isSandboxEnv) {
                    getFromSandboxRoot(tarballPath);
                } else {
                    downloadFromSandboxProxy(tarballPath, tarballSandboxUrl);
                }
            }

            // временная директория используется всегда, кроме запусков под линуксом не в sandbox
            // до этого изменения временная директория использовалась только при запуске под линуксом в sandbox
            // кажется, новый вариант тоже корректный
            Path dir;
            if (getSandboxResourcesRoot() == null && "Linux".equals(System.getProperty("os.name"))) {
                dir = Paths.get("");
            } else {
                dir = createTmpDir();
            }

            Path pathToRedis = dir.resolve(REDIS_SERVER_BINARY_NAME);

            if (!pathToRedis.toFile().exists()) {
                new ProcessBuilder()
                        .directory(tarballPath.toAbsolutePath().getParent().toFile())
                        .command("tar", "-xvf", tarballPath.toAbsolutePath().toString(),
                                "-C", dir.toAbsolutePath().toString(),
                                REDIS_SERVER_BINARY_NAME)
                        .redirectError(tarballPath.resolveSibling("tar.log").toFile())
                        .start().waitFor();
            }

            return pathToRedis.toAbsolutePath().toString();
        } catch (IOException e) {
            throw new RuntimeException("Unable to start tests", e);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Unable to start tests", e);
        }
    }

    private static Path createTmpDir() throws IOException {
        Path dir = Files.createTempDirectory("redis-");
        logger.info("Using temporary directory {} for Redis files", dir.toAbsolutePath());

        Thread shutdownThread = new NamedThreadFactory("DeleteTmpRedisDir")
                .newThread(() -> {
                    Arrays
                            .stream(Objects.requireNonNull(dir.toFile().listFiles()))
                            .forEach(File::delete);
                    dir.toFile().delete();
                });
        Runtime.getRuntime().addShutdownHook(shutdownThread);
        return dir;
    }

    private static void getFromSandboxRoot(Path tarball) throws IOException {
        logger.info("Using resource from sandbox");
        logger.info("Sandbox root is " + getSandboxResourcesRoot());
        logger.info("Sandbox root content: " + Arrays.toString(Paths.get(getSandboxResourcesRoot()).toFile().list()));
        if (!Paths.get(getSandboxResourcesRoot()).resolve(tarball).toFile().exists()) {
            throw new IOException("Failed to fetch redis from sandbox");
        }
    }

    private static void downloadFromSandboxProxy(Path tarball, String sandboxUrl) throws IOException {
        logger.info("Downloading redis");
        URL sandboxProxyUrl = new URL(sandboxUrl);
        var connection = sandboxProxyUrl.openConnection();

        var tokenFile = FileUtils.expandHome("~/.direct-tokens/local_user_sandbox_token");
        if (tokenFile.toFile().exists()) {
            connection.setRequestProperty("Authorization", "OAuth " + FileUtils.readToken(tokenFile));
        }

        Files.copy(connection.getInputStream(), tarball, REPLACE_EXISTING);
    }
}
